'use strict'

var truncate = require('unicode-byte-truncate')
var uuid = require('uuid')

var Span = require('./span')
var symbols = require('../symbols')
var Timer = require('./timer')
var log = require('../logger')
var appendMsg = 'Transaction :'
module.exports = Transaction

/**
 *  * Bundle spans and indivdual transaction in for a single agent
 * transaction.
 * @param {Object} agent 
 * @param {string} name 
 * @param {string} type 
 */
function Transaction (agent, name, type) {
  Object.defineProperty(this, 'name', {
    configurable: true,
    enumerable: true,
    get () {

      return this._customName ||
        this._defaultName ||
        (this.req ? this.req.method + ' unknown route (unnamed)' : 'unnamed')
    },
    set (name) {
      if (this.ended) {
        log.debug(appendMsg , 'tried to set transaction.name on already ended transaction ', { id: this.id })
        return
      }
     
      this._customName = name
    }
  })

  Object.defineProperty(this, 'result', {
    configurable: true,
    enumerable: true,
    get () {
      return this._result
    },
    set (result) {
      if (this.ended) {
        log.debug(appendMsg ,'tried to set transaction.result on already ended transaction ', { id: this.id })
        return
      }
      this._result = result
    }
  })
  this.method = '';
  this.id = uuid.v4() +  '-' + Date.now()
  this._defaultName = name || ''
  this._customName = ''
  this._custom = null
  this.type = type || 'custom'
  this.result = 'success'
  this.spans = []
  this._builtSpans = []
  this._droppedSpans = 0
  this.ended = false
  // this._abortTime = 0
  this._agent = agent
  this._agent._instrumentation.currentTransaction = this

  // Random sampling 
  this.sampled = true;

  log.debug(appendMsg, 'starting new  transaction', { id: this.id })

  this._timer = new Timer()

  this.nodeOrder = '1'
  this.nodeOrderCounter = 0; 
  this._context
}

Transaction.prototype.buildSpan = function () {
  // if (!this.sampled) {
  //   return null
  // }

  if (this.ended) {
   log.debug(appendMsg, 'transaction already ended - cannot build new span ', { id: this.id })
    return null
  }
  // if (this._builtSpans.length >= this._agent._conf.transactionMaxSpans) {
  //   this._droppedSpans++
  //   return null
  // }

  var span = new Span(this)
  this._builtSpans.push(span)
  return span
}


Transaction.prototype.duration = function () {
  if (!this.ended) {
    log.debug(appendMsg, 'tried to call duration() on un-ended transaction', { id: this.id, name: this.name, type: this.type })
    return null
  }

  return this._timer.duration()
}

Transaction.prototype.setDefaultName = function (name) {
  log.debug(appendMsg,'setting default transaction name:', name,  ' id:', { id: this.id })
  this._defaultName = name
}

Transaction.prototype.setContext = function (context) {
  if (!context) return
  this._context = Object.assign(this._context || {}, context)
}

Transaction.prototype.end = function (result) {

  if (this.ended) {
    log.debug(appendMsg, 'tried to call transaction.end() on already ended transaction id : ', { id: this.id })
    return
  }

  if (result !== undefined) {
    this.result = result
  }


  this._builtSpans.forEach(function (span) {
    if (span.ended || !span.started) return
    span.truncate()
  })

  this._timer.end()
  this.ended = true

  var trans = this._agent._instrumentation.currentTransaction

  if (!trans) {
    log.debug(appendMsg, 'WARNING: no currentTransaction found', { current: trans, spans: this.spans.length, id: this.id })
    this.spans = []
  } else if (trans !== this) {
    log.debug(appendMsg, 'WARNING: transaction is out of sync', { spans: this.spans.length, id: this.id, other: trans.id })
    this.spans = []
  }

  this._agent._instrumentation.addEndedTransaction(this)
  log.debug(appendMsg, 'ended ', { id: this.id, type: this.type, result: this.result, name: this.name })
}

Transaction.prototype._recordEndedSpan = function (span) {
  if (this.ended) {
    log.debug(appendMsg, 'Can\'t record ended span after parent transaction have ended - ignoring ', { id: this.id, span: span.name })
    return
  }

  this.spans.push(span)
}
