'use strict'

const os = require('os');
const fs = require('fs');

const logs = [];
let isNoLogs = false;
const notInstalledModules = [];
let logPath = '';
let statusPath = '';
let showError = false;
let forContainerEnv = false;

if (process.argv && process.argv.length && process.argv.indexOf("--no-logs") > -1) {
  isNoLogs = true;
} else {
  if (process.argv && process.argv.length) {
    process.argv.forEach(argv => {
      if (argv.indexOf("--log-path") > -1) {
        logPath = argv.split("=")[1];
      } else if (argv.indexOf("--status-path") > -1) {
        statusPath = argv.split("=")[1];
      } else if (argv.indexOf("--show-error") > -1) {
        showError = true;
      } else if (argv.indexOf("--for-container-env") > -1) {
        forContainerEnv = true;
      }
    });
  }

  if (forContainerEnv) {
    process._rawDebug("\n----------------------------- eG NodeJS Profiler Image build summary starts -----------------------------");
  } else {
    process._rawDebug("Log file Path:" + logPath);
    process._rawDebug("Status file Path:" + statusPath);
  }
}

const packages = require('../package.json');
if (!packages || !packages.dependencies) {
  error("Can't find valid package.json");
  exitCode();
  process.exit();
}

const dependencies = Object.assign(packages.dependencies, packages.optionalDependencies || {});
const dependencyNames = Object.keys(dependencies);
debug(`Going to check ${dependencyNames.length} node modules`);

dependencyNames.forEach(name => {
  try {
    require(name);
    debug(`${name}/${dependencies[name]} is available`);
  } catch (e) {
    if (forContainerEnv) throw e;
    if (showError) console.error(e.message);
    error(`${name}/${dependencies[name]} is not available`, name);
  }
});

exitCode();

function debug(msg) {
  if (isNoLogs) return;
  msg = timestamp() + " - INFO - " + msg;
  console.log(msg);
  logs.push(msg);
}

function error(msg, name) {
  if (name === 'cpu-stats' && os.type && os.type() === 'Windows_NT') return;
  if (isNoLogs) {
    console.log("FAIL");
    process.exit();
  }

  notInstalledModules.push(name);
  msg = timestamp() + " - WARN - " + msg;
  console.error(msg);
  logs.push(msg);
}

function timestamp() {
  const d = new Date();
  const offset = -d.getTimezoneOffset();

  return pad(d.getDate()) + "-" +
    pad(d.getMonth() + 1) + "-" +
    pad(d.getFullYear()) + " " +
    pad(d.getHours()) + ":" +
    pad(d.getMinutes()) + ":" +
    pad(d.getSeconds()) + " " +
    ((offset >= 0 ? "+" : "-") + pad(parseInt(offset / 60)) + ":" + pad(offset % 60))
}

function pad(n) {
  return n < 10 ? "0" + n : n;
}

function exitCode() {
  if (isNoLogs) {
    console.log("PASS");
    process.exit();
  }

  if (forContainerEnv) {
    console.log("SUCCESS: All required node_modules are available");
    console.log("----------------------------- eG NodeJS Profiler Image build summary ends -------------------------------\n\n");
    process.exit();
  }

  if (!logs.length) return;
  let summary = 'Summary:\n    ';
  let status = null;
  const snippetCode = 'You can start monitoring your NodeJS by adding the snippet code as mentioned in the eG Admin Console (Web Application) >> Admin (Top Menu) >> Infrastructure >> Component >> Add/modify >>  Component type to Node.js >> Modify >> MODIFY COMPONENT screen';

  if (!notInstalledModules.length) {
    summary += 'All node modules needed for the eG Node.js monitor is available. ' + snippetCode;
    status = 50;
  } else {
    let mainLibFail = false;
    const notWorkingFn = [];

    notInstalledModules.forEach(name => {
      if (name === 'cpu-stats') {
        notWorkingFn.push("CPU usage of thread");
      } else if (name === 'event-loop-stats') {
        notWorkingFn.push("Some Event Loop metrics");
      } else if (name === 'gc-stats' || name === '@sematext/gc-stats') {
        notWorkingFn.push("Some Garbage Collection metrics");
      } else if (name === 'v8-profiler-next') {
        notWorkingFn.push("Memory and CPU profiler");
      } else if (!packages.optionalDependencies[name]) {
        mainLibFail = true;
      }
    });

    if (mainLibFail) {
      status = 30;
      summary += "The node modules needed for eG Node.js Monitor is not installed properly. So you can't monitor your Node.js application.";
    } else {
      status = 40;
      summary += `Some node modules are not installed properly so Except ${notWorkingFn.join(', ')}, all the other feature of eG Node.js monitor is available. ${snippetCode}`;
    }
  }

  console.log(summary);
  let existingLog = '';
  let existingStatus = '';
  if (!logPath) return;

  try {
    existingLog = fs.readFileSync(logPath, 'utf8');
  } catch (e) {
    console.error(e);
  }

  const logMsg = existingLog + logs.join(' \n') + '\n' + summary;
  fs.writeFile(logPath, logMsg, e => {
    if (e) console.error(e);
    if (!statusPath) return;

    try {
      existingStatus = fs.readFileSync(statusPath, 'utf8') || '';
    } catch (e) {
      console.error(e);
    }

    if (!existingStatus || existingStatus.indexOf('INSTALL_STATUS') === -1) {
      existingStatus += `INSTALL_STATUS=${status}`;
    } else {
      const props = existingStatus.split(/\r?\n/);
      let newLog = '';

      props.forEach(p => {
        if (p.indexOf('INSTALL_STATUS') === -1) {
          newLog += p;
        } else {
          newLog += `\nINSTALL_STATUS=${status}`
        }
      });
      existingStatus = newLog;
    }

    fs.writeFile(statusPath, existingStatus, e1 => {
      if (e1) console.error(e1);
      console.log("logged successfully...!");
    });
  });
}
