const execa = require('execa');
const logger = require('../logger');
const appendMsg = 'pid-port:';

const netstat = async type => {
  try {
    const { stdout } = await execa('netstat', ['-anv', '-p', type]);
    logger.debug(appendMsg, 'netstat ' + type, stdout);
    return stdout;
  } catch (e) {
    logger.error(appendMsg, 'netstat execa', e);
  }
};

const macos = async () => {
  const result = await Promise.all([
    netstat('tcp'),
    netstat('udp'),
  ]);

  return result.join('\n');
};

const linux = async () => {
  try {
    const { stdout } = await execa('ss', ['-tunlp']);
    logger.debug(appendMsg, 'linux ss -tunlp', stdout);
    return stdout;
  } catch (e) {
    logger.error(appendMsg, 'linux execa', e);
  }
};

const win32 = async () => {
  try {
    const { stdout } = await execa('netstat', ['-ano']);
    logger.debug(appendMsg, 'win32 netstat -ano', stdout);
    return stdout;
  } catch (e) {
    logger.error(appendMsg, 'win32 execa', e);
  }
};

const getListFunction = process.platform === 'darwin' ? macos : (process.platform === 'linux' ? linux : win32);
const addressColumn = process.platform === 'darwin' ? 3 : (process.platform === 'linux' ? 4 : 1);
const portColumn = process.platform === 'darwin' ? 8 : (process.platform === 'linux' ? 6 : 4);
const isProtocol = value => /^\s*(tcp|udp)/i.test(value);

const parsePid = pid => {
  if (typeof pid !== 'string') {
    return;
  }

  const { groups } = /(?:^|",|",pid=)(?<pid>\d+)/.exec(pid) || {};
  if (groups) {
    return Number.parseInt(groups.pid, 10);
  }
};

const getPort = (port, list) => {
  const regex = new RegExp(`[.:]${port}$`);
  const foundPort = list.find(line => regex.test(line[addressColumn]));

  if (!foundPort) {
    throw new Error(`Could not find a process that uses port \`${port}\``);
  }

  return parsePid(foundPort[portColumn]);
};

const getList = async () => {
  try {
    const list = await getListFunction();
    if (!list || !list.length) return [];

    return list
      .split('\n')
      .filter(line => isProtocol(line))
      .map(line => line.match(/\S+/g) || []);
  } catch (e) {
    logger.error(appendMsg, 'getList', e);
  }
};

async function portToPid(port) {
  try {
    if (Array.isArray(port)) {
      const list = await getList();
      const tuples = await Promise.all(port.map(port_ => [port_, getPort(port_, list)]));
      return new Map(tuples);
    }

    if (!Number.isInteger(port)) {
      throw new TypeError(`Expected an integer, got ${typeof port}`);
    }

    return getPort(port, await getList());
  } catch (e) {
    logger.error(appendMsg, 'portToPid', e);
  }
}

async function pidToPorts(pid) {
  try {
    if (Array.isArray(pid)) {
      const returnValue = new Map(pid.map(pid_ => [pid_, new Set()]));
      const allPorts = await allPortsWithPid();

      if (allPorts.size) {
        for (const [port, pid_] of allPorts) {
          if (returnValue.has(pid_)) {
            returnValue.get(pid_).add(port);
          }
        }
      }

      return returnValue;
    }

    if (!Number.isInteger(pid)) {
      throw new TypeError(`Expected an integer, got ${typeof pid}`);
    }

    const returnValue = new Set();
    const allPorts = await allPortsWithPid();

    if (allPorts.size) {
      for (const [port, pid_] of allPorts) {
        if (pid_ === pid) {
          returnValue.add(port);
        }
      }
    }

    return returnValue;
  } catch (e) {
    logger.error(appendMsg, 'pidToPorts', e);
  }
}

async function allPortsWithPid() {
  try {
    const list = await getList();
    const returnValue = new Map();

    if (list && list.length) {
      for (const line of list) {
        const { groups } = /[^]*[.:](?<port>\d+)$/.exec(line[addressColumn]) || {};
        if (groups) {
          returnValue.set(Number.parseInt(groups.port, 10), parsePid(line[portColumn]));
        }
      }
    }

    return returnValue;
  } catch (e) {
    logger.error(appendMsg, 'allPortsWithPid', e);
  }
}

exports.portToPid = portToPid;
exports.pidToPorts = pidToPorts;
exports.allPortsWithPid = allPortsWithPid;