'use strict'

const fs = require('fs');
const path = require('path');

const logger = require('./../logger');
const appUtils = require('../app-utils');
const appConstants = require('../app-constants');

const appendMsg = 'Collector:';
require('./socket-client'); // don't remove this line

/**
 * Collector is reponsible for storing data
 * temporarly and pushing the data periodically to the agent 
 * via socket server
 * @param {Object} agent 
 */
function Collector(agent) {
  this.agent = agent;
  this.isStarted = false;
  this.communicator = {};
  this.tempDir = path.normalize(__dirname + '/../../../../temp');
}

function _newCommunicator() {
  if (this.agent.config.isMultithreadCommunicator) {
    try {
      require('worker_threads').threadId;
      const Communicator = require('./communicator-thread');
      return new Communicator(this);
    } catch (e) {
      // the socket comminication will happen in the same thread
      logger.info(appendMsg, "Worker thread is not supported.");
    }
  }

  const CommunicatorS = require('./communicator-standalone');
  return new CommunicatorS(this);
}

Collector.prototype.start = function () {
  if (this.agent._externalTransport && typeof this.agent._externalTransport == 'function') {
    // if any external transport then ignore the socket.
    return;
  }

  this.communicator = _newCommunicator.call(this);
  this.communicator.start(this.agent.config, this.agent.runtime.getInfo());
  this.isStarted = true;
}

/**
 * To add the new data into the collector
 */
Collector.prototype.add = function (data) {
  if (this.agent._externalTransport && typeof this.agent._externalTransport == 'function') {
    this.agent._externalTransport(data);
    return;
  }

  if (!data) return;
  const _self = this;
  if (!this.communicator.add || typeof this.communicator.add != 'function') {
    const timeout = setTimeout(_ => {
      _self.communicator.add(data);
    }, 1000);
    timeout.unref();
    return;
  }

  if (!this.agent.stopedTracing || data.sendImmediately === true) {
    return this.communicator.add(data);
  }
}

Collector.prototype.sendFile = function (type, fileData, options) {
  if (this.agent._externalTransport && typeof this.agent._externalTransport == 'function') {
    this.agent._externalTransport(type, fileData);
    return;
  }

  // if (logger.level === 'silly' && this.agent.config.enable_console_log) {
  //   try {
  //     if (fs.existsSync(this.tempDir)) {
  //       const fileExt = { CPU: 'cpuprofile', MEMORY: 'heapsnapshot' };
  //       const fileName = `${this.tempDir}/${type}_${Date.now()}.${fileExt[type]}`;
  //       fs.writeFile(fileName, fileData, function () { });
  //       logger.info(appendMsg, `The ${type} dump is writen in ${fileName}`);
  //     }
  //   } catch (e) {
  //     logger.error(appendMsg, 'dump store error', e);
  //   }
  // }

  if (appUtils.getStringToBytes(fileData) > appConstants.MAX_DUMP_SIZE) {
    logger.error(appendMsg, `${type} dump size is greater than max size (${appConstants.MAX_DUMP_SIZE / (1024 * 1024)} MB) so it cannot be send`);
    return;
  }

  if (this.agent.stopedTracing) return;
  let nodeDump = options || {};
  if (this.agent.runtime) {
    nodeDump = Object.assign(nodeDump, this.agent.runtime.getInfo());
  }
  nodeDump.dump = fileData
  nodeDump.type = type + '_DUMP';
  nodeDump.processID = process.pid;
  nodeDump.collectedAt = appUtils.formatDate();
  this.communicator.sendFile(type, { nodeDump });
}

Collector.prototype.resetConnection = function () {
  this.communicator.resetConnection();
}

Collector.prototype.onNewComponent = function (newComponentDetails) {
  this.agent.configWatcher.onNewComponent(newComponentDetails);
}

Collector.prototype.setConfig = function (config) {
  this.communicator.setConfig(config);
}

/**
 * update the config dynamically
 */
Collector.prototype.onConfigChange = function (data) {
  this.agent.configWatcher.onConfigChange(data);
}

module.exports = Collector;