'use strict'

const fs = require('fs');
const path = require('path');
const sanitize = require("sanitize-filename");

const appendMsg = 'Cache:';
const logger = require('./logger');
const info = require('./app-info');
const homeDirectory = info.homeDirectory;

try {
  _ensureDirectoryExistence(homeDirectory);
} catch (e) {
  logger.error(appendMsg, 'error in _ensureDirectoryExistence', e);
}

function Cache(fileName) {
  if (!fileName) {
    logger.error(appendMsg, 'fileName is not present.');
    throw "Please provide fileName";
  }

  try {
    this.path = path.normalize(homeDirectory + path.sep + sanitize(fileName));
  } catch (e) {
    logger.error(appendMsg, 'error in createWriteStream', e);
  }
}

module.exports = Cache;

Cache.prototype.write = function (data) {
  try {
    fs.writeFileSync(this.path, data);
  } catch (e) {
    logger.error(appendMsg, 'error in write', e);
  }
}

Cache.prototype.read = function () {
  try {
    if (fs.existsSync(this.path)) {
      return fs.readFileSync(this.path, 'utf-8');
    }
  } catch (e) {
    logger.error(appendMsg, 'error in read', e);
  }
}

Cache.prototype.set = function (key, value) {
  let data = _read.call(this);

  try {
    data[key] = value;
    fs.writeFileSync(this.path, data && JSON.stringify(data, null, 2) || '');
  } catch (e) {
    logger.error(appendMsg, 'error in write', e);
  }
}

Cache.prototype.get = function (key) {
  if (!key) return '';
  const data = _read.call(this);
  return data[key] || '';
}

function _read() {
  let data = {};
  try {
    if (fs.existsSync(this.path)) {
      data = fs.readFileSync(this.path, 'utf-8');
      if (data) data = JSON.parse(data);
    }
  } catch (e) {
    logger.error(appendMsg, 'error in read', e);
  }

  return data || {};
}

Cache.prototype.delete = function () {
  try {
    if (fs.existsSync(this.path)) {
      fs.unlinkSync(this.path);
    }
  } catch (e) {
    logger.error(appendMsg, 'error in delete', e);
  }
}

function _ensureDirectoryExistence(dirname) {
  if (fs.existsSync(dirname)) {
    return true;
  }
  fs.mkdirSync(dirname);
}