'use strict'

const shimmer = require('../shimmer');
const logger = require('../../logger');
const utils = require('../instrumentation-utils');
const safeStringify = require('../../utils/stringify');

let isWrapped = false;
const appendMsg = 'Mongoose:';
const funs = [
  'aggregate',
  'bulkWrite',
  'create',
  'insertMany',
  'save',
];

exports.start = function (mongoose, agent, version, enabled) {
  if (!enabled || !mongoose.Model || !mongoose.Error) return mongoose;

  try {
    isWrapped = true;
    funs.forEach(fn => {
      if (mongoose.Model[fn]) {
        logger.debug(appendMsg, 'wrapping mongoose.Model.' + fn);
        shimmer.wrap(mongoose.Model, fn, wrapFn);
      } else if (mongoose.Model.prototype[fn]) {
        logger.debug(appendMsg, 'wrapping mongoose.Model.prototype.' + fn);
        shimmer.wrap(mongoose.Model.prototype, fn, wrapFn, { prototype: true, model: 'constructor' });
      }
    });

    if (mongoose.Query && mongoose.Query.prototype) {
      logger.debug(appendMsg, 'wrapping mongoose.Query.prototype.exec');
      shimmer.wrap(mongoose.Query.prototype, 'exec', wrapFn, { prototype: true, model: 'model' });
    }

    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }
  return mongoose;

  /** wraping the mongoose to capture the error only. */
  function wrapFn(original, fnName, arg) {
    return function wrappedFn() {
      let _model = this;
      let isSubFnCalled = false;
      const len = arguments.length - 1;
      const cbfn = arguments[len];
      const args = Array.prototype.slice.call(arguments);

      if (arg && arg.prototype === true) {
        _model = arg.model && this[arg.model] || this;
        if (arguments && arguments[1] && typeof arguments[1] === 'function' && arguments[1].name === 'callbackWrapper') {
          // mongoose callback fn name is callbackWrapper so if name is callbackWrapper then it is called internally by mongoose;
          isSubFnCalled = true;
        }
      }

      if (!_model.modelName) return original.apply(this, arguments);
      let names = [_model.baseModelName || _model.modelName, this.op || fnName];
      const name = names.join('.');
      const span = agent.startSpan(name, 'mongodb', 'Mongoose');

      let hasCallback = false;
      if (cbfn && typeof cbfn === 'function') {
        args[len] = wrapCallback(cbfn);
        hasCallback = true;
      }

      let result = null;
      let query = { name };
      const firstArg = args[0];
      if (this._conditions) query.filter = this._conditions;
      if (this._fields) query.fields = this._fields;
      if (this._update) query.update = this._update;
      if (this.options && Object.keys(this.options).length) {
        query = Object.assign(query, this.options);
      }

      if (!query.limit && this.op && this.op.indexOf('findOne') > -1) {
        query.limit = 1;
      }

      if (Object.keys(query).length == 1 && firstArg && typeof firstArg !== 'function') {
        query.filter = firstArg;
      }

      try {
        result = original.apply(this, args);
      } catch (e) {
        setSpanDetails(e);
        throw e;
      }
      if (!hasCallback && typeof result.then === 'function') {
        result.then(resolve, reject);
      }

      return result;

      function resolve() {
        if (span) span.remove();
      }

      function reject(err) {
        setSpanDetails(err);
      }

      function wrapCallback(callback) {
        return function wrappedCallback(err) {
          setSpanDetails(err);
          return callback.apply(this, arguments);
        }
      }

      function setSpanDetails(err) {
        if (!span || !err || isSubFnCalled) return resolve();

        //if (err) process._rawDebug(isMongooseError(err));
        span.options = {};
        const isErrorRecorded = span.captureError(err);
        if (!isErrorRecorded) return resolve();

        const db = _model && _model.db || {};
        span.options.dbName = db.name || '';
        span.options.host = db.host || '';
        span.options.port = db.port || '';
        span.options.entityName = names[0];
        span.options.queryType = names[1];

        if (query) {
          span.options.query = query;
        } else if (result && result._conditions) {
          span.options.query.filter = result._conditions;
        }

        span.options.query = safeStringify(span.options.query);
        const maxChar = agent.getConfig('max_nosql_query_length');
        span.options.query = utils.subString(span.options.query, maxChar);
        span.options.serverType = 'mongodb';
        logger.debug(appendMsg, 'error captured at ' + fnName);
        span.end();
      }
    }
  }

  function isMongooseError(err) {
    const keys = Object.keys(mongoose.Error);
    for (let i = 0; i < keys.length; i++) {
      if (err instanceof mongoose.Error[keys[i]]) return true;
    }
    return false;
  }
}

exports.stop = function (mongoose, version) {
  if (!isWrapped) return;
  funs.forEach(fn => {
    if (mongoose.Model[fn]) {
      logger.debug(appendMsg, 'unwrapping mongoose.Model.' + fn);
      shimmer.unwrap(mongoose.Model, fn);
    } else if (mongoose.Model.prototype[fn]) {
      logger.debug(appendMsg, 'unwrapping mongoose.Model.prototype.' + fn);
      shimmer.unwrap(mongoose.Model.prototype, fn);
    }
  });

  if (mongoose.Query && mongoose.Query.prototype) {
    logger.debug(appendMsg, 'unwrapping mongoose.Query.prototype.exec');
    shimmer.unwrap(mongoose.Query.prototype, 'exec');
  }

  isWrapped = false;
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}