'use strict'

const semver = require('semver');

const shimmer = require('../../shimmer');
const logger = require('../../../logger');
const symbols = require('../../../symbols');
const appConstant = require('../../../app-constants');
const transaction = require('../../transaction-utils');

let isWrapped = false;
const pointCutName = 'amqplib';
const qeGGUID = appConstant.QUEUE_GUID_NAME;
const appendMsg = `RabbitMQ-${pointCutName}:`;
const headerDelimiter = appConstant.MQ_HEADER_DELIMITER;

exports.start = function (amqplib, agent, version, enabled) {
  if (!enabled) return amqplib;
  if (!semver.satisfies(version, '>=0.5')) {
    logger.info(appendMsg, `version ${version} is not supported`);
    return amqplib;
  }

  try {
    if (amqplib.Channel) {
      logger.debug(appendMsg, 'wrapping channel.sendMessage');
      shimmer.wrap(amqplib.Channel.prototype, 'sendMessage', wrapSendMessage);

      logger.debug(appendMsg, 'wrapping channel.sendImmediately');
      shimmer.wrap(amqplib.Channel.prototype, 'sendImmediately', wrapSendMessage);
    }

    if (amqplib.BaseChannel) {
      logger.debug(appendMsg, 'wrapping channel.dispatchMessage');
      shimmer.wrap(amqplib.BaseChannel.prototype, 'dispatchMessage', wrapDispatchMessage);
    }

    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'wrapChannel', e);
  }

  isWrapped = true;
  return amqplib;

  function wrapSendMessage(originalFn) {
    return function wrappedSendMessage() {
      const fields = arguments[0];
      if (!isWrapped || !fields || typeof fields !== 'object') {
        return originalFn.apply(this, arguments);
      }

      const queueName = fields.routingKey || fields.exchange;
      const span = agent.startSpan('', 'rabbitmq', pointCutName);
      if (!span) return originalFn.apply(this, arguments);

      try {
        const url = `${_getUrl.call(this)}//${queueName}`;
        if (fields.headers) fields.headers[qeGGUID] = span.getNeweGGUID() + getNodeOrderSufx(`Async_Listener__${queueName}`, !!fields.exchange, url, 'producer');
        span.name = `RabbitMQ ${url}`;
        span.options = {
          type: 'PRODUCER',
          provider: 'RabbitMQ',
          nodeOrder: span.nodeOrder,
          queue: `${url}`,
          pointcut: 'JMS',
        }
        logger.debug(appendMsg, 'Send Message', {
          queue: span.options.queue,
          headerGoingToSend: fields.headers && fields.headers[qeGGUID]
        });

        let result = null;
        try {
          result = originalFn.apply(this, arguments);
        } catch (err) {
          logger.debug(appendMsg, 'error captured', err.message);
          span.captureError(err);
        }

        span.end();
        return result;
      } catch (e) {
        logger.error(appendMsg, 'wrappedSendMessage', e);
      }
    }
  }

  function getNodeOrderSufx(url, isAppend, mqUrl, type) {
    let suffix = ['', url, 'RabbitMQ', mqUrl, type].join(headerDelimiter);
    suffix = isAppend ? getNicPort(agent) + suffix : suffix;
    return suffix;
  }

  function wrapDispatchMessage(originalFn) {
    return function wrappedDispatchMessage() {
      if (!isWrapped) return originalFn.apply(this, arguments);

      try {
        let trans = null;
        let nodeOrder = null;
        const queueName = arguments[0].routingKey || arguments[0].exchange;
        let url = `${_getUrl.call(this)}//${queueName}`;

        if (!agent.isTransactionFound()) {
          const headers = arguments[1] && arguments[1].properties && arguments[1].properties.headers;
          logger.debug(appendMsg, 'received header', headers);
          const uri = `Async_Listener__${queueName}`;
          const option = transaction.getTransactionWithReqIdentifier(null, agent.config, uri, headers);
          trans = agent.startTransaction(`JMS_ENTRY`, option);

          if (trans && trans.nodeOrder && trans.nodeOrder.indexOf(headerDelimiter) > -1) {
            nodeOrder = trans.nodeOrder.split(headerDelimiter)[0];
            url = trans.nodeOrder.split(headerDelimiter)[3] || url;
            if (nodeOrder.indexOf('[') === -1) nodeOrder += getNicPort(agent);
            trans.nodeOrder = trans.requestIdentifier.nodeOrder = nodeOrder;
          }
        }

        let result = null;
        const span = agent.startSpan(`RabbitMQ ${url}`, 'rabbitmq', pointCutName);

        if (span) {
          span.options = {
            nodeOrder: nodeOrder || '1.1',
            type: 'CONSUMER',
            provider: 'RabbitMQ',
            queue: `${url}`,
            pointcut: 'JMS_ENTRY',
          }
        }

        try {
          result = originalFn.apply(this, arguments);
        } catch (err) {
          logger.debug(appendMsg, 'error captured', err.message);
          if (span) span.captureError(err);
        }

        if (span) span.end();
        if (trans) trans.end();
        return result;
      } catch (e) {
        logger.error(appendMsg, 'wrappedDispatchMessage', e);
      }
    }
  }
}

function getNicPort(agent) {
  const id = agent.config.component_id;
  const guid = agent.config.unique_component_id;
  const randomNum = Math.floor(1000 + Math.random() * 9000); // random 4 digit no
  return `.${id || guid.substr(guid.length - 17)}!~!${randomNum}`;
}

function _getUrl() {
  if (!this.connection || !this.connection[symbols.rabbitMQConnectionDetails]) return '';
  const connDetails = this.connection[symbols.rabbitMQConnectionDetails];
  const username = connDetails.username ? connDetails.username + '@' : '';
  return `${connDetails.protocol}://${username}${connDetails.hostname}:${connDetails.port}`;
}

exports.wrapGet = function (originalFn, name, agent) {
  return function wrappedGet() {
    if (!isWrapped) return originalFn.apply(this, arguments);

    try {
      let trans = null;
      let nodeOrder = null;
      const queueName = arguments[0];
      const uri = `Async_Listener__${queueName}`;
      let url = `${_getUrl.call(this)}//${queueName}`;

      if (!agent.isTransactionFound()) {
        const option = transaction.getTransactionWithReqIdentifier(null, agent.config, uri, {});
        trans = agent.startTransaction(`JMS`, option);
      }

      let result = null;
      const span = agent.startSpan(`RabbitMQ ${url}`, 'rabbitmq', pointCutName);

      if (span) {
        span.options = {
          nodeOrder: nodeOrder || '1.1',
          type: 'CONSUMER',
          provider: 'RabbitMQ',
          queue: `${url}`,
          pointcut: 'JMS',
        }
      }

      try {
        arguments[2] = wrapGetCallback(arguments[2]);
        result = originalFn.apply(this, arguments);
      } catch (err) {
        logger.debug(appendMsg, 'error captured', err.message);
        if (span) span.captureError(err);
      }
      return result;

      function wrapGetCallback(cb) {
        return function wrappedGetCallback(err, msg) {
          const headers = msg && msg.properties && msg.properties.headers;
          logger.debug(appendMsg, 'received header', headers);
          if (err && span) span.captureError(err);
          if (span) span.end();

          if (trans && headers) {
            transaction.getTransactionWithReqIdentifier(trans, agent.config, uri, headers);
            if (trans && trans.nodeOrder && trans.nodeOrder.indexOf(headerDelimiter) > -1) {
              span.options.nodeOrder = trans.nodeOrder.split(headerDelimiter)[0];
              url = trans.nodeOrder.split(headerDelimiter)[3] || url;
              if (span.options.nodeOrder.indexOf('[') === -1) span.options.nodeOrder += getNicPort(agent);
              trans.nodeOrder = trans.requestIdentifier.nodeOrder = span.options.nodeOrder;
            }
          } else if (headers) {
            const egParams = transaction.getGuidAndNodeOrder(headers);
            if (egParams && egParams.length === 2) {
              span.options.nodeOrder = egParams[1] || '1.1';
              if (span.options.nodeOrder.indexOf(headerDelimiter) > -1) {
                span.options.nodeOrder = span.options.nodeOrder.split(headerDelimiter)[0];
              }
              if (span.options.nodeOrder.indexOf('[') === -1) span.options.nodeOrder += getNicPort(agent);
            }
          }

          if (trans) trans.end();
          return cb.apply(this, arguments);
        }
      }
    } catch (e) {
      logger.error(appendMsg, 'wrappedDispatchMessage', e);
    }
  }
}

exports.stop = function (amqplib, version) {
  if (!isWrapped) return;
  try {
    if (amqplib.Channel) {
      shimmer.unwrap(amqplib.Channel.prototype, 'sendMessage');
      shimmer.unwrap(amqplib.Channel.prototype, 'sendImmediately');
    }

    if (amqplib.BaseChannel) {
      shimmer.unwrap(amqplib.BaseChannel.prototype, 'dispatchMessage');
    }
  } catch (e) {
    logger.error(appendMsg, 'stop', e);
  }

  isWrapped = false;
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}