'use strict'

const perf_hooks = require('perf_hooks');

const logger = require('../logger');
const appUtils = require('./../app-utils');
const threads = require('../utils/threads');

const MAX_GC_INFO_SIZE = 1000;
const GC_TYPES = {
  1: 'Scavenge',
  2: 'MarkSweepCompact',
  4: 'IncrementalMarking',
  8: 'ProcessWeakCallbacks',
  15: 'MarkSweepCompact'
};
const appendMsg = 'GC-stats:';
const allowsGCTypes = [1, 2, 15];

function GcStats(infra) {
  _reset.call(this);
  this._infra = infra;
  this.gcObserver = null;
  this.isRunning = false;
  this.isGcSupported = true;
}

module.exports = GcStats;

GcStats.prototype.start = function () {
  if (this.isRunning) return;
  if (threads.isMainThread && this.isGcSupported) {
    try {
      this.gcStats = (require('@sematext/gc-stats'))();
      this.gcStats.on('stats', _onNewStats.bind(this));
      this.isRunning = true;
      logger.info(appendMsg, 'started successfully will be reported using gc-stats module');
    } catch (e) {
      this.isGcSupported = false;
      logger.error(appendMsg, "the gc-stats module couldn't be required", e);
    }
  }

  if (!this.gcStats) {
    try {
      if (!this.gcObserver) {
        this.gcObserver = new perf_hooks.PerformanceObserver((list) => {
          list.getEntries().map(_onNewStatsSummary.bind(this));
        });
      }
      this.gcObserver.observe({ entryTypes: ['gc'], buffered: true });
      this.isRunning = true;
      logger.info(appendMsg, 'started successfully will be reported using perf_hooks.PerformanceObserver');
    } catch (e) {
      logger.error(appendMsg, "PerformanceObserver GC error", e);
    }
  }

  _reset.call(this);
}

function _onNewStatsSummary(stats) {
  if (!stats || !stats.detail || !this.isRunning) return;
  if (threads.isMainThread) {
    _onNewStats.call(this, {
      gctype: stats.detail.kind,
      startTime: stats.startTime,
      pauseMS: appUtils.roundOff(stats.duration),
    });
    return;
  }

  switch (stats.detail.kind) {
    case perf_hooks.constants.NODE_PERFORMANCE_GC_MINOR:
    case perf_hooks.constants.NODE_PERFORMANCE_GC_INCREMENTAL:
    case perf_hooks.constants.NODE_PERFORMANCE_GC_WEAKCB:
      this.gcSummary.noOfMinorOfGc++;
      this.gcSummary.minorGcTime += stats.duration;
      break;
    case perf_hooks.constants.NODE_PERFORMANCE_GC_MAJOR:
      this.gcSummary.noOfFullOfGc++;
      this.gcSummary.fullGcTime += stats.duration;
      break;
  }
}

function _onNewStats(stats) {
  if (!this.isRunning || !stats || !stats.gctype || !stats.startTime || allowsGCTypes.indexOf(stats.gctype) === -1) return;
  this.gc = this.gc || [];

  if (this.gc.length >= MAX_GC_INFO_SIZE) {
    this._infra.sendinfra({
      gc: this.getStats()
    });
  }

  const currentDate = new Date();
  let starttime = Math.round(stats.startTime / 100000000000);
  starttime = currentDate.getTime() - starttime;
  const gc = {
    starttime,
    duration: stats.pauseMS || 0,
    type: GC_TYPES[stats.gctype] || stats.gctype.toString(),
  };

  if (stats.after) gc.heapAfterGc = stats.after.usedHeapSize;//Number of bytes in use 
  if (stats.before) gc.heapBeforeGc = stats.before.usedHeapSize;
  this.gc.push(gc);
}

GcStats.prototype.getStats = function () {
  if (!this.gc.length) return [];
  let tempGc = null;

  try {
    tempGc = this.gc.slice();
    _reset.call(this);
  } catch (e) {
    logger.error(appendMsg, "getStats error", e, this.gc);
  }
  return tempGc;
}

GcStats.prototype.getSummary = function () {
  if (!this.gcObserver) return;
  const newSummary = Object.assign({}, this.gcSummary);
  newSummary.fullGcTime = appUtils.roundOff(newSummary.fullGcTime);
  newSummary.minorGcTime = appUtils.roundOff(newSummary.minorGcTime);
  _reset.call(this);
  return newSummary;
}

function _reset() {
  this.gc = [];
  this.gcSummary = {
    fullGcTime: 0,
    minorGcTime: 0,
    noOfFullOfGc: 0,
    noOfMinorOfGc: 0,
  };
}

GcStats.prototype.stop = function () {
  if ((!this.gcStats && !this.gcObserver) || !this.isRunning) return;

  if (this.gcStats) {
    this.gcStats.off();
  } else if (this.gcObserver) {
    this.gcObserver.disconnect();
  }

  _reset.call(this);
  this.isRunning = false;
  logger.info(appendMsg, 'Stopped successfully...!');
}