'use strict'

const semver = require('semver');
const perf_hooks = require('perf_hooks');

const logger = require('./../logger');
const appUtils = require('./../app-utils');
const threads = require('../utils/threads');

const appendMsg = 'Event-Loop:';

class EventLoop {
  constructor() {
    this.lastElu = null;
    this.performance = null;
    this.eventLoopLag = null;
    this.eventLoopStats = null;
    this.perfHooksELMonitor = null;
    this.eventLoopStatsErr = false;
  }

  start(lagInterval) {
    this._requirePerformance();
    this._requireEventLoopStats();

    if (threads.isMainThread) {
      this._requireEventLoopLag(lagInterval);
    }
    logger.info(appendMsg, `started successfully...!`);
  }

  restart(lagInterval) {
    this._stopEventLoopLag();
    this._requireEventLoopLag(lagInterval);
  }

  _requirePerformance() {
    if (this.performance) {
      this.lastElu = this.performance.eventLoopUtilization();
      return;
    }

    if (!semver.gte(process.version, '12.19.0')) {
      logger.error(appendMsg, 'perf_hooks:performance eventLoopUtilization will be supported after NodeJS 12.19.0...!');
      return;
    }

    try {
      this.performance = perf_hooks.performance;
      this.lastElu = this.performance.eventLoopUtilization();
      logger.info(appendMsg, 'perf_hooks:performance npm required successfully...');
    } catch (e) {
      logger.error(appendMsg, "perf_hooks:performance couldn't be required", e);
    }
  }

  _requireEventLoopStats() {
    if (this.eventLoopStats) {
      this.eventLoopStats.sense();
      return;
    }

    if (this.perfHooksELMonitor) {
      this.perfHooksELMonitor.enable();
      this.perfHooksELMonitor.reset();
      return;
    }

    if (this.eventLoopStatsErr) return;

    if (threads.isMainThread) {
      if (!semver.gte(process.version, '8.0.0')) {
        logger.error(appendMsg, 'event-loop-stats will be supported after NodeJS 8.0.0...!');
        return;
      }

      try {
        this.eventLoopStats = require('event-loop-stats');
        logger.info(appendMsg, 'event-loop-stats npm required successfully...');
        return;
      } catch (e) {
        this.eventLoopStatsErr = true;
        logger.error(appendMsg, "Event-loop-stats couldn't be required", e);
      }
    }

    if (!semver.gte(process.version, '11.10.0')) {
      logger.info(appendMsg, 'perf_hooks monitorEventLoopDelay will be supported after NodeJS 11.10.0');
      return;
    }

    try {
      this.perfHooksELMonitor = perf_hooks.monitorEventLoopDelay({ resolution: 10 });
      this.perfHooksELMonitor.enable();
    } catch (err) {
      this.eventLoopStatsErr = true;
      logger.error(appendMsg, "perf_hooks monitorEventLoopDelay couldn't be required", err);
    }
  }

  _requireEventLoopLag(lagInterval) {
    if ('number' != typeof lagInterval || !lagInterval) {
      logger.error(appendMsg, "Event-loop-lag Interval should be a valid number");
      return;
    }

    try {
      this.eventLoopLag = require('event-loop-lag')(lagInterval);
      logger.info(appendMsg, `event-loop-stats npm required successfully with lag interval ${lagInterval / 1000}`);
    } catch (e) {
      logger.error(appendMsg, "Event-loop-stats couldn't be required", e);
    }
  }

  getStats() {
    const stats = this._getStats() || {};
    this._setLag(stats);
    this._setElu(stats);
    return stats;
  }

  _setElu(stats) {
    if (!this.performance) return;
    const elu = this._getElu();
    appUtils.setEventLoopUtilization(elu, stats);
  }

  _getElu() {
    const currentElu = this.performance.eventLoopUtilization();
    const elu = this.performance.eventLoopUtilization(currentElu, this.lastElu);
    this.lastElu = currentElu;
    return elu;
  }

  _setLag(stats) {
    if (!this.eventLoopLag) return;
    stats.lag = Math.round(this.eventLoopLag() * 100) / 100;
  }

  _getStats() {
    if (!this.eventLoopStats) {
      if (!this.perfHooksELMonitor) return;
      const stats = {};
      stats.mean = appUtils.roundOff(this.perfHooksELMonitor.mean / 1e6);

      if (threads.isMainThread) {
        stats.min = appUtils.roundOff(this.perfHooksELMonitor.min / 1e6);
        stats.max = appUtils.roundOff(this.perfHooksELMonitor.max / 1e6);
      }

      this.perfHooksELMonitor.reset();
      return stats;
    }

    const stats = this.eventLoopStats.sense();
    if (!stats) {
      logger.error(appendMsg, 'stats not found', stats);
      return;
    }

    const avg = stats.sum / stats.num;
    if (stats.min > stats.max || avg > stats.max || avg < stats.min) {
      if (stats.min > stats.max || avg < stats.min) stats.min = 0;
      if (avg > stats.max) stats.max = avg;
      logger.error(appendMsg, 'Invalid stats', stats);
    }

    return stats;
  }

  stop() {
    if (this.perfHooksELMonitor) {
      this.perfHooksELMonitor.disable();
    }

    this._stopEventLoopLag();
    logger.info(appendMsg, 'Stopped successfully...!');
  }

  _stopEventLoopLag() {
    if (this.eventLoopLag) {
      try {
        const path = require.resolve('event-loop-lag');
        if (require.cache[path]) delete require.cache[path];
        this.eventLoopLag = null;
      } catch (e) {
        logger.error(appendMsg, 'delete require.cache error', stats);
      }
    }
  }
}

module.exports = EventLoop;