'use strict'

const Path = require('path');
const { BroadcastChannel } = require('worker_threads');

const logger = require('../../logger');
const symbols = require('../../symbols');
const appInfo = require('../../app-info');
const appUtils = require('../../app-utils');
const appConstants = require('../../app-constants');
const Timer = require('../../instrumentation/timer');
const shimmer = require('../../instrumentation/shimmer');

let workers = {};
let isWrapped = false;
let broadcastChannel = null;
let transportChannel = null;
const MAX_WT_TO_MONITOR = 50;
const appendMsg = 'Worker Threads:';

exports.start = function (worker_threads, agent, version, enabled) {
  if (!enabled) return worker_threads;

  const workerThreadAgentPath = Path.normalize(__dirname + '/../../worker-threads/worker-metrics/index.js');
  isWrapped = true;

  try {
    logger.debug(appendMsg, 'wrapping worker_threads.Worker');
    shimmer.wrap(worker_threads, 'Worker', wrapWorker);
    broadcastChannel = new BroadcastChannel(appConstants.WT_BROADCAST_CHANNEL);
    transportChannel = new BroadcastChannel(appConstants.WT_TRANSPORT_CHANNEL);
    broadcastChannel.unref();
    transportChannel.unref();

    transportChannel.onmessage = (event) => {
      if (!event || !event.data) return;
      const threadId = event.data.threadId;
      if (!threadId) return;

      if (event.data.workerThreadInfra) {
        const worker = workers[threadId];
        if (!worker || !worker[symbols.threadInfo]) return;

        Object.keys(event.data.workerThreadInfra).forEach(key => {
          worker[symbols.threadInfo][key] = event.data.workerThreadInfra[key];
        });
        logger.info(appendMsg, 'Infra metrics received from thread ' + threadId);
      } else if (event.data.dump && event.data.type) {
        // agent.collector.sendFile(event.data.type, event.data.dump, { threadId });
        logger.info(appendMsg, event.data.type + ' dump received from thread ' + threadId);
      }
    };

    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'channel error', e);
  }

  agent.infraManager.getWorkerThreadMetrics = function (cb) {
    if (agent.getConfig('enable_wtm_from_inside_thread')) {
      collectWorkerThreadInfra();
      // lets wait 10 sec for worker to send their metrics
      const timeout = setTimeout(() => {
        sendMetrics(cb);
      }, 1000 * 10);
      timeout.unref();
    } else {
      sendMetrics(cb);
    }
  }

  return worker_threads;

  function wrapWorker(original) {
    return function wrappedWorker(filename, options) {
      if (!isWrapped || getWorkersLength() > MAX_WT_TO_MONITOR) return new original(filename, options);
      logger.info(appendMsg, 'On new worker thread, enable_wtm_from_inside_thread: ' + agent.getConfig('enable_wtm_from_inside_thread'));

      if (agent.getConfig('enable_wtm_from_inside_thread')) {
        options = options || {};
        options.env = options.env || {};
        options.execArgv = options.execArgv || [];
        options.execArgv.push(`--require`);
        options.execArgv.push(workerThreadAgentPath);
        const params = {};
        params.workerPath = filename;
        params.config = agent.config;
        params.logDirectory = appInfo.logDirectory;
        params.homeDirectory = appInfo.homeDirectory;
        options.env.EG_NODE_PARAM = JSON.stringify(params);
      }

      const worker = new original(filename, options);
      _onNewWorker(worker, filename);
      return worker;
    }
  }

  function _onNewWorker(worker, filename) {
    if (!worker) return;
    worker[symbols.threadInfo] = {
      timer: new Timer(),
      workerPath: filename,
      threadId: worker.threadId,
    };
    workers[worker.threadId] = worker;
    const performance = worker.performance;

    if (!agent.getConfig('enable_wtm_from_inside_thread') && performance && performance.eventLoopUtilization) {
      worker[symbols.threadInfo].lastELU = performance.eventLoopUtilization();
    }

    worker.on('error', (err) => {
      agent.captureError(err, {
        threadId: worker.threadId && worker.threadId.toString()
      });
    });

    worker.on('exit', () => {
      const threadInfo = worker[symbols.threadInfo];
      if (!threadInfo) return;

      threadInfo.timer.end();
      const execTime = threadInfo.timer.durationInMs();
      const threadId = threadInfo && threadInfo.threadId || -1;
      delete workers[threadInfo.threadId];
      logger.info(appendMsg, `Worker thread is exit, ID: ${threadId}, execTime: ${execTime}, active thread cnt: ${getWorkersLength()}`);
    });

    logger.info(appendMsg, `New worker thread is created, ID: ${worker.threadId}, active thread cnt: ${getWorkersLength()}, worker path: ${filename}`);
  }

  function getWorkersLength() {
    return Object.keys(workers).length;
  }

  function collectWorkerThreadInfra() {
    if (!isWrapped || !getWorkersLength() || !broadcastChannel) return;

    broadcastChannel.postMessage({
      config: agent.config,
      action: appConstants.WT_INFRA_BROADCAST_MSG,
    });
    logger.info(appendMsg, 'Broadcast Message has been sent to all worker thread to collect infra metrics');
  }

  function sendMetrics(cb) {
    const allMetrics = [];
    Object.keys(workers).forEach(id => {
      const worker = workers[id];
      const threadInfo = worker[symbols.threadInfo];
      const metrics = {
        threadId: id,
        workerPath: threadInfo.workerPath,
        uptime: threadInfo.timer.runningTime(),
      };

      if (agent.getConfig('enable_wtm_from_inside_thread')) {
        metrics.cpuUsage = threadInfo.cpuUsage;
        metrics.eventloop = threadInfo.eventloop;
        metrics.gcSummary = threadInfo.gcSummary;
        metrics.memoryUsage = threadInfo.memoryUsage;
      } else {
        metrics.eventloop = getEventLoopUtilization(worker);
      }
      allMetrics.push(metrics);
    });

    cb(null, allMetrics);
  }

  function getEventLoopUtilization(worker) {
    const threadInfo = worker[symbols.threadInfo];
    const performance = worker.performance;
    if (!performance || !performance.eventLoopUtilization) return;

    const currentElu = performance.eventLoopUtilization();
    const elu = performance.eventLoopUtilization(currentElu, threadInfo.lastELU);
    threadInfo.lastELU = currentElu;
    const eventloop = {};
    appUtils.setEventLoopUtilization(elu, eventloop);
    return eventloop;
  }
}

exports.stop = function (worker_threads, version) {
  if (!isWrapped) return;
  shimmer.unwrap(worker_threads, 'Worker');

  if (broadcastChannel) {
    broadcastChannel.postMessage({ action: appConstants.STOP_WTM_MSG });
    broadcastChannel.close();
    broadcastChannel = null;
  }

  if (transportChannel) {
    transportChannel.close();
    transportChannel = null;
  }

  isWrapped = false;
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}