'use strict'

const logger = require('../logger');

const maxProfileDuration = 5000; // ms
const appendMsg = 'cpu-profiler:';
const cpuProfileName = 'eg-node-cpu-profile';

function CPUProfiler(agent) {
  this.agent = agent;
  this.isRunning = false;

  try {
    this.v8Profiler = require("v8-profiler-next");
    logger.info(appendMsg, 'v8-profiler-next npm required successfully...');
  } catch (e) {
    logger.error(appendMsg, "v8-profiler-next couldn't be required", e);
  }
}

module.exports = CPUProfiler;

CPUProfiler.prototype.start = function () {
  if (!this.v8Profiler) {
    logger.debug(appendMsg, "v8-profiler-next is not found, so can't do profiling");
    return;
  }

  if (this.isRunning) {
    logger.debug(appendMsg, "don't profile more than one at a time");
    return;
  }

  logger.info(appendMsg, `started successfully...!`);
  this.isRunning = true;
}

CPUProfiler.prototype.collectAndSendProfile = function () {
  if (this.isProfiling) return;

  const _self = this;
  this.isProfiling = true;
  let duration = this.agent.getConfig('cpu_profiling_duration');
  this.doProfile(duration, (e, data) => {
    _self.isProfiling = false;
    if (e) return;
    _processDump.call(_self, data);
  });
}

CPUProfiler.prototype.doProfile = function (duration, cb) {
  if (!this.v8Profiler) return;
  
  try {
    const _self = this;
    duration = duration > maxProfileDuration ? maxProfileDuration : duration;
    logger.debug(appendMsg, `Started profiler for ${duration} seconds`);
    this.v8Profiler.startProfiling(cpuProfileName, true);

    this.timeoutPromise = setTimeout(function () {
      try {
        _self.timeoutPromise = null;
        const snapshot = _self.v8Profiler.stopProfiling(cpuProfileName);

        snapshot.export(function (err, result) {
          _deleteProfiler.call(_self, snapshot);
          if (err) {
            logger.error(appendMsg, "Error while exporting snapshot", err);
            if (cb) cb(true);
            return;
          }

          logger.debug(appendMsg, `Profiling completed`);
          if (cb) cb(null, result);
        });
      } catch (e1) {
        _deleteProfiler.call(_self, snapshot);
        logger.error(appendMsg, "Error while profiling", e1);
        if (cb) cb(true);
      }
    }, duration);
    this.timeoutPromise.unref();
  } catch (e) {
    logger.error(appendMsg, "Error while profiling", e);
    if (cb) cb(true);
  }
}

function _processDump(data) {
  if (!data) return;

  try {
    this.agent.collector.sendFile('CPU', data);
  } catch (e) {
    logger.error(appendMsg, "Error while proccessing dump", e);
  }
}

function _deleteProfiler(snapshot) {
  if (snapshot) snapshot.delete();
  this.v8Profiler.deleteAllProfiles();
}

CPUProfiler.prototype.stop = function () {
  if (!this.v8Profiler || !this.isRunning) return;

  if (this.timeoutPromise) {
    clearTimeout(this.timeoutPromise);
    const snapshot = this.v8Profiler.stopProfiling();
    _deleteProfiler.call(_self, snapshot);
  }

  if (this.intervalPromise) clearInterval(this.intervalPromise);
  this.isRunning = false;
  this.isProfiling = false;
  this.intervalPromise = null;
  logger.info(appendMsg, 'stopped successfully');
}