'use strict'

const uuid = require('uuid');

const Span = require('./span');
const Timer = require('./timer');
const logger = require('../logger');
const appUtils = require('../app-utils');
const appConstant = require('../app-constants');

module.exports = Transaction
const appendMsg = 'Transaction:'
const dbs = appConstant.SQL_DBS.concat(appConstant.NOSQL_DBS);

/**
 *  * Bundle spans and indivdual transaction in for a single agent
 * transaction.
 * @param {Object} agent 
 * @param {string} name 
 * @param {string} type 
 */
function Transaction(agent, type, option) {
  this.method = '';
  this.id = uuid.v4() + '-' + Date.now()
  this.type = type || 'custom'
  this.spans = []
  this.errors = [];
  this._builtSpans = []
  this._droppedSpans = 0
  this.ended = false
  // this._abortTime = 0
  this._agent = agent

  this.userContext = null;
  this.egEnabledInReqOrigin = false;
  this._timer = new Timer();
  this.nodeOrder = '1'
  this.nodeOrderCounter = 0;
  this._context
  this.sqlTransaction = 0;
  this.dbQuery = {};
  this.fastDBQuery = {};
  this.slowExceedDBQuery = {};
  this.errorExceedDBQuery = {};
  this.errTranceCnt = 0;
  this.errStackTraceCnt = 0;
  this.errFqenCaptureCnt = 0;

  if (option) {
    Object.keys(option).forEach(key => {
      this[key] = option[key];
    });
  }

  if (this.requestIdentifier) {
    const reqInd = this.requestIdentifier;
    const threshold_config = agent.config.threshold_config;

    if (!reqInd.eg_guid) {
      reqInd.eg_guid = this.id;
    }

    if (reqInd.uri && threshold_config[reqInd.uri]) {
      reqInd.slowUrlThreshold = threshold_config[reqInd.uri][1];
      reqInd.stallUrlThreshold = threshold_config[reqInd.uri][0];
    }

    reqInd.slowUrlThreshold = reqInd.slowUrlThreshold || agent.getConfig('slow_url_threshold');
    reqInd.stallUrlThreshold = reqInd.stallUrlThreshold || agent.getConfig('stalled_url_threshold');

    this.stallTimer = setTimeout(_ => {
      logger.debug(appendMsg, 'transaction is taking more time, so going to end this.')
      this.end();
    }, reqInd.stallUrlThreshold);
    this.stallTimer.unref();
  }

  logger.debug(appendMsg, 'starting new transaction', { id: this.id, name: this.name, type: this.type })
}

Transaction.prototype._getInfo = function () {
  return {
    id: this.id,
    name: this.name,
    type: this.type,
    uri: this.requestIdentifier && this.requestIdentifier.uri
  }
}

Transaction.prototype.buildSpan = function (type) {
  if (this.ended) {
    logger.debug(appendMsg, 'transaction is already ended - cannot build new span', type);
    return null;
  }

  if (type && dbs.indexOf(type) > -1) {
    this.sqlTransaction++;
    const sqlLimit = this._agent.getConfig('max_db_trace_limit_per_transaction');
    if (sqlLimit > 0 && sqlLimit < this.sqlTransaction) {
      return null;
    }
  }

  // if (this._builtSpans.length >= this._agent._conf.transactionMaxSpans) {
  //   this._droppedSpans++
  //   return null
  // }

  const span = new Span(this);
  span.type = type;
  this._builtSpans.push(span)
  return span
}

Transaction.prototype.removeSpan = function (span) {
  const index = this.getSpanIndex(span);
  if (index > -1) {
    this._builtSpans.splice(index, 1);
  }
}

Transaction.prototype.getSpanIndex = function (span) {
  for (let i = 0; i < this._builtSpans.length; i++) {
    if (span.id === this._builtSpans[i].id) return i;
  }

  return -1
}

Transaction.prototype.duration = function () {
  if (!this.ended) {
    logger.debug(appendMsg, 'tried to call duration() on un-ended transaction', this.id);
    return null;
  }

  return this._timer.duration();
}

Transaction.prototype.end = function (result) {
  if (this.ended) {
    logger.debug(appendMsg, 'transaction is already ended transaction id: ', this._getInfo())
    return
  }

  try {
    logger.debug(appendMsg, 'going to end', this._getInfo());
    if (this.stallTimer) clearTimeout(this.stallTimer);


    this._builtSpans.forEach(function (span) {
      if (span.ended || !span.started) return;
      span.truncate();
    })

    this._timer.end()
    this.ended = true
    setUriAndQueryStringForExpress.call(this);
    this.req = null;
    this._agent._instrumentation.addEndedTransaction(this);
    logger.debug(appendMsg, 'ended', this._getInfo());
  } catch (e) {
    logger.debug(appendMsg, 'end error ', e);
  }
}

function setUriAndQueryStringForExpress() {
  if (!this.req) return;
  if (this.req.query && !appUtils.isEmptyObj(this.req.query)) {
    this.requestIdentifier.queryString = this.req.query;
  } else if (this.req.params && !appUtils.isEmptyObj(this.req.params)) {
    const params = Object.assign({}, this.req.params);
    // for static URL's the query string is like  { '0': 'url' } in express. to avoid this below code
    if ('/' + params[0] == this.req.url || params[0] == this.req.url) {
      delete params[0];
    }

    if (!appUtils.isEmptyObj(params))
      this.requestIdentifier.queryString = params;
  }

  //For expressjs pattern path
  if (this.req.route) {
    //To handle '/api/customer/:id' pattern,
    let path = '';

    if (this.req.route.path) {
      if (typeof this.req.route.path !== 'string') {
        path = this.req.route.path.toString();
      } else {
        path = this.req.route.path;
      }
    } else if (this.req.route.regexp && this.req.route.regexp.source) {
      path = this.req.route.regexp.source;
    }

    if (path && path[path.length - 1] === '?') {
      path = path.substring(0, path.length - 1);
    }

    if (path && path.indexOf('/:') > -1) {
      this.requestIdentifier.name = path;
      const uri = this.req.protocol + '://' + this.req.get('host');
      if (this.req.baseUrl && this.req.baseUrl !== uri && this.req.baseUrl.substring(0, 4) !== 'http' && this.req.baseUrl.substring(0, 5) !== 'https') {
        this.requestIdentifier.name = this.req.baseUrl + this.requestIdentifier.uri;
      }
    }
  }
}
Transaction.prototype._recordEndedSpan = function (span) {
  if (this.ended) {
    logger.debug(appendMsg, `Can't record ended span after parent transaction have ended - ignoring`, span._getInfo());
    return;
  }

  this.removeSpan(span);
  if (span.isNotNeeded === true) return;
  if (_isNeedToIgnore.call(this, span)) return;

  this.spans.push({
    uid: span.uid,
    name: span.name,
    type: span.type,
    ended: span.ended,
    options: span.options,
    startTime: span._timer.start,
    duration: span.duration,
  });
}

function _isNeedToIgnore(span) {
  if (span.uid) {
    // to avoid duplicates
    const dup = this.spans.find(e => e.uid == span.uid);
    if (dup) return true;
  }

  if (dbs.indexOf(span.type) === -1 || !span.options) {
    span.duration = span._timer.durationInMs();
    return;
  }

  const key = `${span.options.host}:${span.options.port}-${span.options.dbName}`;
  const minTraceTime = this._agent.getConfig('min_dbq_exec_track_time');
  const maxQueryCnt = this._agent.getConfig('max_query_to_track_per_db');
  const maxErrorQueryCnt = this._agent.getConfig('max_error_query_to_track_per_db');
  span.duration = span._timer.durationInMs(minTraceTime < 1);

  // if fast query and no error then no need to track all details
  if (span.duration < minTraceTime && !span.options.error) {
    if (!this.fastDBQuery[key]) {
      this.fastDBQuery[key] = _getNewSqlEle(span, 'FASTSQL');
    }

    this.fastDBQuery[key].resTime += span.duration;
    this.fastDBQuery[key].query++;
    logger.debug(appendMsg, "It is a fast query so ignoring it.", span._getInfo());
    return true;
  }

  if (!this.dbQuery[key]) {
    this.dbQuery[key] = {
      queryTracked: 0,
      slowQueryTracked: 0,
      errorQueryTracked: 0
    }
  }

  // if the tracked query is more than the max limit then it will be ignored
  if (maxQueryCnt <= this.dbQuery[key].queryTracked) {
    let obj = null;
    if (span.options.error) {
      obj = this.errorExceedDBQuery[key] || _getNewSqlEle(span, 'ErrorExceedSql');
      this.errorExceedDBQuery[key] = obj;
      logger.debug(appendMsg, "error query exceeds so ignoring it.", span._getInfo());
    } else {
      obj = this.slowExceedDBQuery[key] || _getNewSqlEle(span, 'SlowExceedSql');
      this.slowExceedDBQuery[key] = obj;
      logger.debug(appendMsg, "slow query exceeds so ignoring it.", span._getInfo());
    }

    obj.resTime += span.duration;
    obj.query++;
    return true;
  }

  // if the tracked error query is more than max error limit then it will be ignored
  if (span.options.error && maxErrorQueryCnt <= this.dbQuery[key].errorQueryTracked) {
    if (!this.errorExceedDBQuery[key]) {
      this.errorExceedDBQuery[key] = _getNewSqlEle(span, 'ErrorExceedSql');
    }

    this.errorExceedDBQuery[key].resTime += span.duration;
    this.errorExceedDBQuery[key].query++;
    logger.debug(appendMsg, "error query exceeds so ignoring it.", span._getInfo());
    return true;
  }


  this.dbQuery[key].queryTracked++;
  if (span.duration >= minTraceTime) {
    this.dbQuery[key].slowQueryTracked++;
  }
  if (span.options.error) {
    this.dbQuery[key].errorQueryTracked++;
  }
}

function _getNewSqlEle(span, nodeOrder) {
  return {
    host: span.options.host,
    port: span.options.port,
    dbName: span.options.dbName,
    serverType: span.options.serverType,
    resTime: 0,
    nodeOrder,
    query: 0,
  }
}

Transaction.prototype.getGuid = function () {
  return this.id + appConstant.GUID_SEPARATOR + this.nodeOrder;
}

Transaction.prototype.getDataForRUM = function () {
  const id = this.userContext && this.userContext.sessionID || '';
  const username = this.userContext && (this.userContext.username || this.userContext.email) || '';
  const encodedSessionId = Buffer.from(id + "#|#" + username).toString('base64');
  return [this.getGuid(), this.getServerTime() || 1, this._agent.config.unique_component_id, encodedSessionId].join('||');
}

Transaction.prototype.getServerTime = function () {
  return this._timer.durationInMs(false, true);
}