'use strict'

const logger = require('../logger');
const appUtils = require('../app-utils');
const appConstant = require('../app-constants');

const appendMsg = 'Transaction:';
const eGGUID = appConstant.GUID_NAME;
const qeGGUID = appConstant.QUEUE_GUID_NAME;
const GUID_SEPARATOR = appConstant.GUID_SEPARATOR;

exports.getTransactionWithReqIdentifier = function (trans, config, url, headers, method, connection) {
  trans = trans || { requestIdentifier: {} };
  try {
    headers = headers || {};
    trans.requestIdentifier = getParams(config, url, headers, method, connection)
    trans.requestIdentifier.reqTime = appUtils.formatDate();

    //Re assign the guid & node order if not exists
    if (trans.requestIdentifier.eg_guid) {
      trans.id = trans.requestIdentifier.eg_guid;
      trans.egEnabledInReqOrigin = true;
    } else if (trans.id) {
      trans.requestIdentifier.eg_guid = trans.id;
    }

    if (trans.requestIdentifier.nodeOrder) {
      trans.nodeOrder = trans.requestIdentifier.nodeOrder;
    } else {
      trans.nodeOrder = '1';
      trans.requestIdentifier.nodeOrder = '1';
    }
  } catch (e) {
    logger.error(appendMsg, 'getTransactionWithReqIdentifier', e);
  }
  return trans;
}

function getParams(config, url, headers, method, connection) {
  const requestIdentifier = {};
  requestIdentifier.uri = url;
  requestIdentifier.queryString = '-';

  try {
    if (requestIdentifier.uri.indexOf('?') > -1) {
      //To remove querystring from the url
      const splittedURL = url.split('?');
      requestIdentifier.uri = splittedURL[0];
      requestIdentifier.queryString = splittedURL[1];
    }

    //https://www.npmjs.com/package/request-ip
    let clientIP = connection && connection.remoteAddress;
    if (clientIP && clientIP.substr(0, 7) == "::ffff:") {
      clientIP = clientIP.substr(7);
    }

    if (clientIP === '::1') clientIP = '127.0.0.1';
    requestIdentifier.remote_host = clientIP || '-';
    requestIdentifier.remoteAddressHeaderName = config.remote_header_address;

    if (config.http_headers != true) {
      const hname = config.remote_header_address || 'x-forwarded-for';
      requestIdentifier.remoteAddressHeaderValue = headers[hname] || headers[hname.toLowerCase()] || '-';
    }

    const egParams = _getGuidAndNodeOrder(headers);

    if (egParams && egParams.length === 2) {
      logger.debug(appendMsg, 'http request contains eGGUID and Node Order, ', egParams.join(','));
      requestIdentifier.eg_guid = egParams[0];
      requestIdentifier.nodeOrder = egParams[1];
    } else {
      logger.debug(appendMsg, 'request not contain the eG Guid');
    }

    requestIdentifier.referer = headers['Referer'] || headers['referer'] || '-';
    requestIdentifier.method = method || '-';
    requestIdentifier.headers = _getHeader(config, headers);
    requestIdentifier.cookies = _getCookies(config, headers);
    requestIdentifier.userAgent = headers['user-agent'] || '-';
    requestIdentifier.resTime = null;
    requestIdentifier.reqType = 'Web';
    requestIdentifier.isRequestRedirect = false;
  } catch (e) {
    logger.error(appendMsg, 'error while collecting http request identifier', e)
  }

  return requestIdentifier;
}

function _getGuidAndNodeOrder(headers) {
  if (!headers) return;
  //HTTP headers are being converted to lower case 
  const tempHeader = {};
  Object.keys(headers).forEach(key => {
    tempHeader[key.toString().toLowerCase()] = headers[key];
  });

  const guid = tempHeader[eGGUID] || headers[qeGGUID];
  if (!guid) return;
  return guid.split(GUID_SEPARATOR);
}
exports.getGuidAndNodeOrder = _getGuidAndNodeOrder;

function _getCookies(config, headers) {
  let cookie = headers['cookie'] || headers['Cookie'];
  if (!cookie || config.http_cookies != true) return '-';
  return cookie;
}

function _getHeader(config, headers) {
  if (!headers || config.http_headers != true) return '-';
  const keyToIgnore = ['referer', 'user-agent', eGGUID, 'cookie'];
  const newHeader = {};
  let isAdded = false;

  Object.keys(headers).forEach(key => {
    if (keyToIgnore.indexOf(key.toLowerCase()) > -1) return;
    newHeader[key] = headers[key];
    isAdded = true;
  });

  if (!isAdded) return '-';
  return JSON.stringify(newHeader);
}
