'use strict'

const semver = require('semver');

const shimmer = require('../shimmer');
const logger = require('../../logger');
const symbols = require('../../symbols');
const utils = require('../instrumentation-utils');

let isWrapped = false;
const appendMsg = 'Mssql-tedious:';
const isWrappedSym = symbols.isWrappedSym;

exports.start = function (tedious, agent, version, enabled) {
  if (!enabled || tedious[isWrappedSym]) return tedious;
  if (!semver.satisfies(version, '>=1.0.0')) {
    logger.debug(appendMsg, `version ${version} not supported`);
    return tedious;
  }

  const ins = agent._instrumentation;
  try {
    isWrapped = true;
    logger.debug(appendMsg, 'wrapping tedious.Connection.prototype.connect');
    shimmer.wrap(tedious.Connection.prototype, 'connect', wrapConnect);

    logger.debug(appendMsg, 'wrapping tedious.Connection.prototype.makeRequest');
    shimmer.wrap(tedious.Connection.prototype, 'makeRequest', wrapMakeRequest);

    tedious[isWrappedSym] = true;
    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }
  return tedious;

  function wrapConnect(original) {
    return function wrappedConnect() {
      const origCallback = arguments[0];
      arguments[0] = ins.bindFunction(function tracedCallback(err) {
        if (err) {
          logger.debug(appendMsg, 'error captured');
          agent.captureError(err);
        }

        if (origCallback && typeof origCallback === 'function') {
          return origCallback.apply(this, arguments);
        }
      });
      return original.apply(this, arguments);
    }
  }

  function wrapMakeRequest(original) {
    return function wrappedMakeRequest(request, packetType, payload) {
      try {
        if (!request.parametersByName || request[isWrappedSym]) return original.apply(this, arguments);
        const span = agent.startSpan('', 'mssql', 'tedious');
        if (!span) return original.apply(this, arguments);

        request[isWrappedSym] = true;
        const params = request.parametersByName;
        let query = params.statement || params.stmt || request.sqlTextOrProcedure || {};
        query = typeof query === 'string' ? query : query.value || '';
        query = query || (payload && payload.toString && payload.toString()) || '';
        span.name = utils.getSpanNameFromQuery(query);
        span.options = { serverType: 'sqlserver', query };

        // extract hostname and port from connection config
        if (typeof this.config === 'object') {
          span.options.host = this.config.server;
          if (this.config.options) {
            span.options.port = this.config.options.port;
            span.options.dbName = this.config.options.database;
          }
        }

        const origCallback = request.userCallback;
        request.userCallback = ins.bindFunction(function tracedCallback(err) {
          if (err) {
            logger.debug(appendMsg, 'error captured');
            span.captureError(err);
          }

          span.end();
          if (origCallback && typeof origCallback === 'function') {
            return origCallback.apply(this, arguments);
          }
        });
      } catch (e) {
        logger.error(appendMsg, 'wrappedMakeRequest error', e);
      }
      return original.apply(this, arguments);
    }
  }
}

exports.stop = function (tedious, version) {
  if (!isWrapped) return;
  isWrapped = false;
  shimmer.unwrap(tedious.Connection.prototype, 'connect');
  shimmer.unwrap(tedious.Connection.prototype, 'makeRequest');
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}