'use strict'

const logger = require('./logger');
const appUtils = require('./app-utils');

const appendMsg = 'Config-Watcher:';

function ConfigWatcher(agent) {
  this.agent = agent;
}

module.exports = ConfigWatcher;

ConfigWatcher.prototype.onNewComponent = function (newComponentDetails) {
  if (!newComponentDetails) return;
  const nickPort = newComponentDetails.nickPort;
  const uniqueComponentId = newComponentDetails.guid;

  if (!this.agent.config.unique_component_id) {
    this.agent.config.component_id = nickPort;
    this.agent.config.unique_component_id = uniqueComponentId;
    this.agent.collector.setConfig(this.agent.config);
    logger.info(appendMsg, `New component details received ${uniqueComponentId} and nick port is ${nickPort}`);
    this.agent.metadata.set('NICK_PORT', nickPort);
    this.agent.metadata.set('GUID', uniqueComponentId);

    if (this.agent.getConfig('enable_btm')) {
      logger.info(appendMsg, 'BTM instrumentation is starting after the discovery');
      this.agent._instrumentation.start();
    }

    if (this.agent.getConfig('enable_inframetrics')) {
      logger.info(appendMsg, 'Inframetrics is starting after the discovery');
      this.agent.infraManager.start();
    }

    return;
  }

  if (this.agent.config.unique_component_id !== newComponentDetails.uniqueComponentId) {
    logger.error(appendMsg, `New component details received which is differ from old, Old: ${this.agent.config.unique_component_id}, New: ${newComponentDetails.uniqueComponentId}`);
  } else {
    logger.info(appendMsg, `Same component details is received: ${this.agent.config.unique_component_id}`);
  }
}

const numberFields = [
  'cpu_profiling_duration',
  'inframetrics_interval',
  'max_url_segments',
  'max_exception_trace_line_count',
  'max_exception_trace_limit',
  'exception_fqen_capture_limit',
  'max_exce_stack_trace_capture_limit',
  'slow_url_threshold',
  'stalled_url_threshold',
  'min_dbq_exec_track_time',
  'max_query_to_track_per_db',
  'max_error_query_to_track_per_db',
  'max_nosql_query_length',
  'log_file_count',
  'log_file_size',
  'socket_timeout',
  'btm_interval',
];
const booleanFields = [
  'enable_btm',
  'enable_inframetrics',
  'enable_infra_exceptions',
  'enable_cpu_metrics',
  'enable_memory_metrics',
  'enable_event_loop_metrics',
  'enable_gc_metrics',
  'enable_uptime_metrics',
  'enable_cpu_profiling',
  'take_memory_snapshot',
  'http_headers',
  'http_cookies',
  'http_status_code',
  'NodeTransTest',
  'enable_http_status_code_test',
  'enable_worker_thread_metrics',
  'enable_rum_injection',
  'enable_rum_btm_transaction_mapping',
];

/**
 * update the config dynamically
 */
ConfigWatcher.prototype.onConfigChange = function (data) {
  const _self = this;
  const config = this.agent.config;
  try {
    /** socket configs */
    _setIfValueChanged('socket_timeout', 'timeout', e => {
      if (e) return;
      _self.communicator.setConfig(config);
      _self.communicator.setTimeout();
    });

    /** Log configs */
    // if (data.log_level) data.log_level = data.log_level.toString().toLowerCase();
    // _setIfValueChanged('log_level', e => {
    //   if (e) return;
    //   logger.setLevel(config.log_level);
    // });
    // _setIfValueChanged('log_file_count', e => {
    //   if (e) return;
    //   logger.setConfig(config);
    // });
    // _setIfValueChanged('log_file_size', e => {
    //   if (e) return;
    //   logger.setConfig(config);
    // });

    /** BTM configs */
    _setIfValueChanged('enable_btm', 'NodeTransTest', e => {
      if (e) return;
      if (config.enable_btm) {
        _self.agent._instrumentation.start();
        logger.info(appendMsg, 'BTM instrumentation is started by manager');
      } else {
        _self.agent._instrumentation.stop();
        logger.info(appendMsg, 'BTM instrumentation is stopped by manager');
      }
    });

    _setIfValueChanged('btm_interval', 'NodeTransTestFrequency');
    _setIfValueChanged('max_url_segments');
    _setIfValueChanged('slow_url_threshold');
    _setIfValueChanged('stalled_url_threshold');
    _setIfValueChanged('chars_to_exclude');
    _setIfValueChanged('http_headers');
    _setIfValueChanged('http_cookies');
    _setIfValueChanged('http_status_code');
    _setIfValueChanged('remote_header_address', 'Miscellaneous.REMOTE_HEADER_ADDRESS');

    _setIfValueChanged('enable_http_status_code_test', 'HttpStatusCodeTest', e => {
      if (e) return;
      if (config.enable_http_status_code_test) {
        _self.agent._instrumentation.httpStatusCodeTest.start();
        logger.info(appendMsg, 'Http Status Code Test is started by manager');
      } else {
        _self.agent._instrumentation.httpStatusCodeTest.stop();
        logger.info(appendMsg, 'Http Status Code Test is stopped by manager');
      }
    });

    _setIfValueChanged('excluded_patterns', e => {
      if (e) return;
      appUtils.setBlackListedUrls(config);
    });

    /** DB Configs */
    _setIfValueChanged('min_dbq_exec_track_time', 'sql_exec_cutoff');
    _setIfValueChanged('max_query_to_track_per_db', 'max_sql_query_count');
    _setIfValueChanged('max_error_query_to_track_per_db', 'max_sql_error_query_count');
    _setIfValueChanged('max_nosql_query_length', 'max_mongodb_query_length');

    /** RUM Config */
    _setIfValueChanged('rum_snippet', 'RUM.RUM_code_snippet');
    _setIfValueChanged('enable_rum_injection', 'RUM.Inject_rum_script_via_btm');
    _setIfValueChanged('enable_rum_btm_transaction_mapping', 'RUM.Drop_RUM_BTM_Integration_Cookie');

    /** Exception Configs */
    _setIfValueChanged('max_exception_trace_line_count');
    _setIfValueChanged('max_exception_trace_limit');
    _setIfValueChanged('exception_fqen_capture_limit');
    _setIfValueChanged('max_exce_stack_trace_capture_limit');
    _setIfValueChanged('ignore_exceptions', e => {
      if (e) return;
      config.ignored_exceptions = _getValidExceptionName(config.ignore_exceptions);
    });
    _setIfValueChanged('include_exceptions', e => {
      if (e) return;
      config.included_exceptions = _getValidExceptionName(config.include_exceptions);
    });

    /** Infra configs */
    _setIfValueChanged('enable_infra_exceptions', 'NodeErrorTest');
    _setIfValueChanged('enable_cpu_metrics', 'NodeCPUTest', e => {
      if (e || !config.enable_cpu_metrics) return;
      _self.agent.infraManager.cpu.start();
    });

    _setIfValueChanged('enable_event_loop_metrics', 'NodeELTest', e => {
      if (e) return;
      if (config.enable_event_loop_metrics) {
        _self.agent.infraManager.eventLoop.start();
      } else {
        _self.agent.infraManager.eventLoop.stop();
      }
    });

    _setIfValueChanged('enable_gc_metrics', 'NodeGCtest', e => {
      if (e) return;
      if (config.enable_gc_metrics) {
        _self.agent.infraManager.gcStats.start();
      } else {
        _self.agent.infraManager.gcStats.stop();
      }
    });

    _setIfValueChanged('enable_memory_metrics', 'NodeMemoryTest');
    _setIfValueChanged('enable_uptime_metrics', 'NodeUptimeTest');
    _setIfValueChanged('cpu_profiling_duration', 'cpu_profiling_duration_(ms)');

    _setIfValueChanged('enable_cpu_profiling', e => {
      if (e) return;
      if (config.enable_cpu_profiling) {
        this.agent.infraManager.cpuProfile.start();
      } else {
        this.agent.infraManager.cpuProfile.stop();
      }
    });

    _setIfValueChanged('take_memory_snapshot', e => {
      if (e) return;
      if (config.take_memory_snapshot === true) {
        this.agent.infraManager.memoryProfile.takeSnapshot();
        config.take_memory_snapshot = false;
      }
    });

    _setIfValueChanged('enable_worker_thread_metrics', 'NodeWrkerThreadTest', e => {
      if (e) return;
      if (config.enable_worker_thread_metrics) {
        this.agent.threadInstrumentation.start();
      } else {
        this.agent.threadInstrumentation.stop();
      }
    });

    _startOrStopInfraManager.call(this, config);
    _setIfInfraTestFrequencyIsChanged(data, config, e => {
      if (e) return;
      _self.agent.infraManager.restart();
    });

    _setThresholdConfig(data.thresholdConfig, config);
  } catch (e) {
    logger.error(appendMsg, 'changeConfig :', e);
  }

  function _setIfValueChanged(name, nameInAgent, cb) {
    if (typeof nameInAgent === 'function') {
      return _setIfValueChanged(name, null, nameInAgent);
    }

    nameInAgent = nameInAgent || name;
    if (!data.hasOwnProperty(nameInAgent)) {
      if (cb && typeof cb === 'function') cb(true);
      return false;
    }

    if (booleanFields.indexOf(name) > -1) {
      data[nameInAgent] = _parseBoolean(data[nameInAgent]);
    } else if (numberFields.indexOf(name) > -1) {
      data[nameInAgent] = parseInt(data[nameInAgent]);
    }

    if (config[name] !== data[nameInAgent]) {
      logger.info(appendMsg, `${name} is changed by manager, old value=${config[name]}, new Value=${data[nameInAgent]}`);
      config[name] = data[nameInAgent];
      if (cb && typeof cb === 'function') cb();
      return true;
    }

    if (cb && typeof cb === 'function') cb(true);
    return false;
  }
}

function _startOrStopInfraManager(config) {
  if (config.enable_inframetrics && !config.enable_infra_exceptions && !config.enable_event_loop_metrics && !config.enable_gc_metrics && !config.enable_uptime_metrics && !config.enable_cpu_metrics && !config.enable_memory_metrics && !config.enable_worker_thread_metrics) {
    config.enable_inframetrics = false;
    this.agent.infraManager.stop();
    logger.info(appendMsg, 'All inframetrics is stopped by manager');
  }

  if (!config.enable_inframetrics && (config.enable_infra_exceptions || config.enable_event_loop_metrics || config.enable_gc_metrics || config.enable_uptime_metrics || config.enable_cpu_metrics || config.enable_memory_metrics || config.enable_worker_thread_metrics)) {
    config.enable_inframetrics = true;
    this.agent.infraManager.start();
    logger.info(appendMsg, 'Some inframetrics is started by manager');
  }
}

function _setIfInfraTestFrequencyIsChanged(data, config, cb) {
  if (!config.enable_inframetrics) return cb(true);
  const freqKeys = [
    'NodeCPUTestFrequency',
    'NodeMemoryTestFrequency',
    'NodeELTestFrequency',
    'NodeGCtestFrequency',
    'NodeUptimeTestFrequency',
    'NodeErrorTestFrequency',
    'NodeWorkerThreadTestFrequency',
  ];

  for (let i = 0; i < freqKeys.length; i++) {
    const key = freqKeys[i];
    if (data.hasOwnProperty(key) && parseInt(data[key]) !== config.inframetrics_interval) {
      logger.info(appendMsg, `inframetrics_interval is changed by manager, old value=${config.inframetrics_interval}, new Value=${data[key]}`);
      config.inframetrics_interval = parseInt(data[key]);
      cb();
      return true;
    }
  }

  cb(true);
  return false;
}

function _getValidExceptionName(names) {
  if (names && names.toString().toLowerCase() != 'none') {
    return names.split(',').map(e => e.toString().trim());
  }
  return [];
}

function _parseBoolean(val) {
  return val === true || val === 'true';
}

function _setThresholdConfig(thresholdConfig, config) {
  if (!thresholdConfig) return;

  Object.keys(thresholdConfig).forEach(key => {
    const keys = key.split('Thresh4B>');
    if (!keys[1] || keys[1] === 'Summary') return;
    const value = thresholdConfig[key].split('/').map(val => parseInt(val));

    if (config.threshold_config[keys[1]] != value) {
      logger.info(appendMsg, `threshold config for ${keys[1]} is changed by manager, old value=${config.threshold_config[keys[1]] || ''}, new Value=${value || ''}`);
      if (value) config.threshold_config[keys[1]] = value;
      else delete config.threshold_config[keys[1]];
    }
  });
}