'use strict'

const endOfStream = require('end-of-stream');

const logger = require('../logger');
const appUtils = require('../app-utils');
const appConstant = require('../app-constants');
const transaction = require('./transaction-utils');
const resTransformer = require('./http-res-transformer');
const instrumentationUtils = require('./instrumentation-utils');

const appendMsg = 'Http-shared:';
const transactionForResponse = new WeakMap();

exports.instrumentRequest = function (agent, moduleName) {
  return function (orig) {
    return function onRequest(event, req, res) {
      if (event === 'request') onNewRequest(agent, moduleName, req, res);
      return orig.apply(this, arguments);
    }
  }
}

function onNewRequest(agent, moduleName, req, res) {
  let trans = null;
  const ins = agent._instrumentation;

  try {
    logger.debug(appendMsg, 'New incoming', moduleName, 'request call url:', req.url);

    //CHECK url is Blacklisted
    if (isUrlExcluded(agent, req.url)) {
      logger.debug(appendMsg, 'ignoring excluded url', req.url);
      return _ignoreTheRequest();
    }

    const option = getRequestIdentifier(agent, req.url, req.headers, req.method, req.connection && req.connection.remoteAddress);
    option.method = req.method;
    trans = agent.startTransaction(`${moduleName}:inbound`, option);
    if (!trans) return _ignoreTheRequest();

    trans.req = req;
    resTransformer.transform(agent, req, res, moduleName);
    transactionForResponse.set(res, trans);
    ins.bindEmitter(req);
    ins.bindEmitter(res);

    endOfStream(res, _endOfStream); // End of stream
  } catch (e) {
    logger.error(appendMsg, moduleName + " error ", e);
  }

  function _endOfStream(err) {
    try {
      if (trans.ended) return;

      if (!err) {
        endTranaction(trans, res);
        return;
      }

      // Handle case where res.end is called after an error occurred on the
      // stream (e.g. if the underlying socket was prematurely closed)
      const end = res.end;
      res.end = function () {
        const result = end.apply(this, arguments)
        endTranaction(trans, res);
        return result;
      }

      // if error in endOfStream and the trans is not ended even after 60 sec then EG will end the transaction
      const timeout = setTimeout(function stalledTransTimeout() {
        endTranaction(trans, res);
      }, 60 * 1000);
      timeout.unref();
    } catch (er) {
      deleteTransaction(res);
      logger.error(appendMsg, moduleName + " endOfStream error ", er);
    }
  }

  function _ignoreTheRequest() {
    // don't leak previous transaction
    agent._instrumentation.supersedeWithEmptyRunContext();
    // for testing perpose, we are calling the fn. don't remove this.
    if (agent._externalTransport) agent._externalTransport();
  }
}
exports.onNewRequest = onNewRequest;

function endTranaction(trans, res) {
  deleteTransaction(res);
  if (trans.ended) return;
  trans.requestIdentifier.statusCode = res.statusCode || null;
  trans.requestIdentifier.statusMessage = res.statusMessage || "";

  if (appConstant.REDIRECT_STATUS_CODE.indexOf(res.statusCode) > -1 && instrumentationUtils.getResHeader.call(res, 'Location')) {
    trans.requestIdentifier.isRequestRedirect = true;
  }

  logger.debug(appendMsg, `call ended with status code: ${res.statusCode}, id: ${trans.id}`);
  trans.end();
}

function getTransaction(res) {
  try {
    return transactionForResponse.get(res);
  } catch (e) { }
}
exports.getTransaction = getTransaction;

function deleteTransaction(res) {
  try {
    transactionForResponse.delete(res);
  } catch (e) { }
}

/**
 * Number of options 
 * Filter by status code
 * filter by Filetype
 * filter by regex pattern 
 * filter urls
 * filter by req header contains req.headers['user-agent'] 
 * @param {} url 
 * @param {*} statusCode 
 */

function isUrlExcluded(agent, url) {
  const splittedURL = url.split('?');
  const urlPath = splittedURL[0];
  const blackListedUrls = agent.config.black_listed_urls;

  try {
    if (blackListedUrls && blackListedUrls instanceof RegExp && blackListedUrls.test(urlPath)) {
      return true;
    }
  } catch (e) {
    logger.error(appendMsg, 'isUrlExcluded fn throw error', e, 'URL:', url)
  }

  return false;
}

exports.isUrlExcluded = isUrlExcluded;

function getRequestIdentifier(agent, url, headers, method, remoteAddress) {
  return transaction.getTransactionWithReqIdentifier(null, agent.config, url, headers, method, remoteAddress);
}
exports.getRequestIdentifier = getRequestIdentifier;
