'use strict'

const logger = require('../../logger');
const Cpu = require('../../infra/cpu');
const Memory = require('../../infra/memory');
const GcStats = require('../../infra/gc-stats');
const EventLoop = require('../../infra/eventloop');
const CpuProfile = require('../../infra/cpu-profiler');

const appendMsg = 'Worker Threads Infra:';

class WorkerThreadInfra {
  constructor(agent) {
    this.agent = agent;
    this.cpu = new Cpu();
    this.memory = new Memory();
    this.gcStats = new GcStats();
    this.eventLoop = new EventLoop();
    this.cpuProfile = new CpuProfile();
  }

  start() {
    try {
      this.cpu.start();
      this.memory.start();
      this.gcStats.start();
      this.eventLoop.start();
    } catch (err) {
      logger.consoleError(appendMsg, 'start error', err);
    }

    logger.info(appendMsg, 'started...!');
  }

  sendMetrics() {
    const data = {};
    const config = this.agent.config;

    if (config.enable_cpu_metrics === true) {
      data.cpuUsage = this.cpu.getUsage();
    }

    if (config.enable_memory_metrics === true) {
      data.memoryUsage = this.memory.getUsage();
    }

    if (config.enable_event_loop_metrics === true) {
      data.eventloop = this.eventLoop.getStats();
    }

    if (config.enable_gc_metrics === true) {
      data.gcSummary = this.gcStats.getSummary();
    }

    this.agent.sendPayload({ workerThreadInfra: data });

    if (config.enable_cpu_profiling === true) {
      this.cpuProfile.doProfile(config.cpu_profiling_duration, (e, data) => {
        this.agent.sendPayload({ dump: data, type: 'CPU' });
      });
    }
  }

  stop() {
    this.gcStats.stop();
    this.eventLoop.stop();
    logger.info(appendMsg, 'stopped...!');
  }
}

module.exports = WorkerThreadInfra;