'use strict'

const path = require('path');
const cluster = require('cluster');

const appendMsg = 'Runtime:'
const logger = require('./logger');

function Runtime(agent) {
  this._agent = agent;
  this.type = null;
  this.workerId = null;
  this.workerCount = 0;
  this.isRunning = false;
  this.isMaster = !!(cluster.isMaster || cluster.isPrimary);
}

module.exports = Runtime;

Runtime.prototype.start = function (cb) {
  let isCbTriggered = false;
  this.isRunning = true;
  const _self = this;

  try {
    if (this.isMaster) {
      this.mode = 'STANDALONE';
      cluster.on('exit', _workerExit);
      cluster.on('online', _workerOnline);
      _cb();
      return;
    }

    if (!cluster.worker) {
      this.mode = 'STANDALONE';
      return;
    }

    this.mode = 'CLUSTER';
    this.workerId = cluster.worker.id;
    global.workerId = this.workerId;

    process.on('message', (data) => {
      _self.type = 'core';
      if (data && data.workerCount) {
        _self.workerCount = data.workerCount;
      }
      _cb();
    });

    const pm2 = _getPm2();
    pm2.list((err, list) => {
      _onWorkerDiscovery();

      if (err) {
        logger.error(appendMsg, `pm2 list error`, err);
        return;
      }
      if (!list || !list.length) return;

      const currentCluster = list.find(e => process.pid == e.pid);
      if (!currentCluster) return;

      list = list.filter(e => {
        return e && e.name && e.name === currentCluster.name;
      });

      _self.type = 'pm2';
      _self.workerCount = list.length || 1;
      _self.workerId = currentCluster.pm_id;
      global.workerId = _self.workerId;
      _cb();
    });
  } catch (e) {
    logger.error(appendMsg, e);
  }

  function _getPm2() {
    if (!process.argv || !process.argv.length) return;
    let pm2Path = null;
    const pm2Finder = path.sep + 'pm2' + path.sep;

    for (let i = 0; i < process.argv.length; i++) {
      const pos = process.argv[i].split(pm2Finder);
      if (pos.length != 2) continue;
      pm2Path = pos[0] + path.sep + 'pm2';
      break;
    }

    if (!pm2Path) return;
    let pm2 = null;

    try {
      pm2 = require(pm2Path);
    } catch (e) {
      logger.error(appendMsg, 'Not able to find pm2', e);
    }
    return pm2;
  }

  function _cb() {
    if (isCbTriggered) return;
    isCbTriggered = true;
    if (cb) cb();
    logger.info(appendMsg, `started with isMaster: ${_self.isMaster}, Mode: ${_self.mode}, workerCount: ${_self.workerCount}, id: ${_self.workerId || null}`);
  }

  function _onWorkerDiscovery() {
    setTimeout(_ => {
      if (_self.workerCount) return;

      logger.error(appendMsg, `Can't find the workerCount. The eg-node-monitor is not enabled for the master node process`);
      _self.workerCount = 1;
      _cb();
    }, 1000 * 10);
  }

  function _workerExit(worker, code, signal) {
    _self._agent.infraManager.workerExitInfo(worker, code, signal);
    _self.workerCount = Object.keys(cluster.workers || {}).length;
    logger.debug(appendMsg, `worker ${worker.process.pid} died, code ${code}, signal ${signal}`);

    // send updated count to other worker
    Object.keys(cluster.workers || {}).forEach(id => {
      cluster.workers[id].send({ workerCount: _self.workerCount });
    });
  }

  function _workerOnline(worker) {
    if (!_self.workerCount) {
      _self.type = 'core';
      _self.workerCount = Object.keys(cluster.workers || {}).length;
      _self.mode = _self.workerCount > 0 ? 'CLUSTER' : 'STANDALONE';
      logger.info(appendMsg, `Runtime started with isMaster: ${_self.isMaster}, Mode: ${_self.mode}, workerCount: ${_self.workerCount}`);
    }

    worker.send({ workerCount: _self.workerCount });
  }
}

Runtime.prototype.getInfo = function () {
  const info = {
    mode: this.mode,
  }

  if (this.mode === 'CLUSTER') {
    info.isMaster = this.isMaster;
    if (info.isMaster) {
      info.workerCount = Object.keys(cluster.workers || {}).length;
      info.workerId = null;
    } else if (cluster.worker) {
      info.workerId = this.workerId;
      info.workerCount = this.workerCount;
    }
  }

  return info;
}