'use strict'

const semver = require('semver');
const EventEmitter = require('events');

const shimmer = require('../shimmer');
const logger = require('../../logger');
const utils = require('../instrumentation-utils');

let isWrapped = false;
const appendMsg = 'Mysql:';
const connectionWraped = Symbol('eGconnectionWraped')

exports.start = function (mysql, agent, version, enabled) {
  if (!enabled) return mysql;

  try {
    isWrapped = true;
    if (!semver.satisfies(version, '^2.0.0')) {
      logger.info(appendMsg, 'version ', version, ' not supported');
      return mysql;
    }

    logger.debug(appendMsg, 'wrapping mysql.createPool');
    shimmer.wrap(mysql, 'createPool', wrapCreatePool);

    logger.debug(appendMsg, 'wrapping mysql.createPoolCluster');
    shimmer.wrap(mysql, 'createPoolCluster', wrapCreatePoolCluster);

    logger.debug(appendMsg, 'wrapping mysql.createConnection');
    shimmer.wrap(mysql, 'createConnection', wrapCreateConnection);

    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }
  return mysql

  function wrapCreateConnection(original) {
    return function wrappedCreateConnection() {
      var connection = original.apply(this, arguments)

      wrapQueryable(connection, 'connection')

      return connection
    }
  }

  function wrapCreatePool(original) {
    return function wrappedCreatePool() {
      var pool = original.apply(this, arguments)

      logger.debug(appendMsg, 'wrapping mysql pool.getConnection')
      shimmer.wrap(pool, 'getConnection', wrapGetConnection)

      return pool
    }
  }

  function wrapCreatePoolCluster(original) {
    return function wrappedCreatePoolCluster() {
      var cluster = original.apply(this, arguments)

      logger.debug(appendMsg, 'wrapping mysql cluster.of')
      shimmer.wrap(cluster, 'of', function wrapOf(original) {
        return function wrappedOf() {
          var ofCluster = original.apply(this, arguments)

          logger.debug(appendMsg, 'wrapping mysql cluster of.getConnection')
          shimmer.wrap(ofCluster, 'getConnection', wrapGetConnection)

          return ofCluster
        }
      })

      return cluster
    }
  }

  function wrapGetConnection(original) {
    return function wrappedGetConnection() {
      var cb = arguments[0]

      if (typeof cb === 'function') {
        arguments[0] = agent.bindFunction(function wrapedCallback(err, connection) {
          if (connection && enabled) {
            wrapQueryable(connection, 'getConnection() > connection');
          }
          return cb.apply(this, arguments)
        })
      }

      return original.apply(this, arguments)
    }
  }

  /**
  number of ways to query in Mysql library
  query(query, CB)   ('SELECT * FROM `books` WHERE `author` = "David"',  CB)
  query(query, value, CB)  ('SELECT * FROM `books` WHERE `author` = ?', ['David'], CB) 
  query(Object,CB)  ({sql: 'SELECT * FROM `books` WHERE `author` = ?',values: ['David']}, CB)
  query(Object,CB)  ({sql: 'SELECT * FROM `books` WHERE `author` = ?'}, ['David'], CB)

  For mysql pool the callback is passed via  _callback: [Function], 
 */

  function wrapQueryable(obj, objType) {
    if (obj[connectionWraped]) return
    // obj[connectionWraped] = true;
    logger.debug(appendMsg, 'wrapping mysql', objType, 'query fn',)
    shimmer.wrap(obj, 'query', wrapQuery)

    function wrapQuery(original) {
      return function wrappedQuery(sql, values, cb) {
        if (!isWrapped) return original.apply(this, arguments);
        const span = agent.startSpan('', 'mysql', 'MySql');
        if (!span) return original.apply(this, arguments);

        const id = span.transaction.id;
        let hasCallback = false;
        let sqlStr;
        logger.debug(appendMsg, 'new mysql query call:' + objType + '.query(), id:', id);

        switch (typeof sql) {
          case 'string':
            sqlStr = sql
            break
          case 'object':
            if (typeof sql._callback === 'function') {
              sql._callback = wrapCallback(sql._callback)
            }
            sqlStr = sql.sql
            break
          case 'function':
            arguments[0] = wrapCallback(sql)
            break
        }

        if (sqlStr) {
          logger.debug(appendMsg, 'mysql query statement', {
            sql: sqlStr,
            id
          })
          span.name = utils.getSpanNameFromQuery(sqlStr);
        }

        if (typeof values === 'function') {
          arguments[1] = wrapCallback(values)
        } else if (typeof cb === 'function') {
          arguments[2] = wrapCallback(cb)
        }

        var params = getParameters(this);
        if (sqlStr) params.query = sqlStr;
        span.options = params;

        var result = original.apply(this, arguments);

        //If callback not found we need use emitter to end span
        //Note:- this is for if the call back is not found then we need to use the emit function

        if (result && !hasCallback && result instanceof EventEmitter) {
          shimmer.wrap(result, 'emit', function (original) {
            return function wrappedEmit(event, err) {
              switch (event) {
                case 'error':
                  if (err) {
                    logger.debug(appendMsg, 'error captured');
                    span.captureError(err);
                  }
                case 'end':
                  span.end()
              }
              return original.apply(this, arguments)
            }
          });
        }

        return result

        function wrapCallback(cb) {
          hasCallback = true;
          return agent.bindFunction(function wrappedCallback(err) {

            //Capture error 
            if (err) {
              logger.debug(appendMsg, 'error captured');
              span.captureError(err);
            }
            span.end()
            return cb.apply(this, arguments)
          });
        }
      }
    }

    function getParameters(sqlObj) {
      var params = {}
      if (sqlObj.config) {
        params.host = sqlObj.config.host
        params.port = sqlObj.config.port
        params.dbName = sqlObj.config.database
        params.serverType = 'mysql'
      }

      return params;
    }

  }
}

exports.stop = function (mysql, version) {
  if (!isWrapped) return;
  shimmer.unwrap(mysql, 'createPool')
  shimmer.unwrap(mysql, 'createPoolCluster')
  shimmer.unwrap(mysql, 'createConnection')
  isWrapped = false;
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}