'use strict'

const uuid = require('uuid');
const moment = require('moment');
const queryString = require('query-string');

const logger = require('../logger');
const appUtils = require('../app-utils');
const threads = require('../utils/threads');
const appConstants = require('../app-constants');

function getSpanNameFromQuery(query) {
  if (!query) return 'Unknow query';
  if (typeof query === 'object') query = query.toString();
  let queryType = query.split(' ');
  queryType = queryType.filter(e => e && !!e.trim()).map(e => e.trim());
  if (queryType.length <= 2) return query;
  queryType = queryType[0] + ' query';
  return subString(queryType, 50);
}

function subString(str, maxChhar) {
  if (str.length <= maxChhar) return str;
  return str.substring(0, maxChhar - 3) + '...';
}

exports.subString = subString;
exports.getSpanNameFromQuery = getSpanNameFromQuery;

/**
 * Filter the httpoutbound and mysql calls from the transaction 
 * and attached those values to the transaction event_data
 * @param {} transaction 
 */
exports.getBtmPayload = function (transaction) {
  let dataToSend = transaction.requestIdentifier && Object.assign({}, transaction.requestIdentifier) || {};
  dataToSend = Object.assign(dataToSend, this._agent.runtime.getInfo());
  dataToSend.resTime = transaction._timer.durationInMs();
  dataToSend.callTrace = [];

  if (transaction.type === 'external') {
    dataToSend.name = transaction.name;
  }

  let errorProcessedIndex = 0;
  let name = transaction.method ? transaction.method + ' ' : '';
  const callTrace = {
    name: name + transaction.name,
    type: transaction.type,
    startTime: transaction._timer.start,
    duration: transaction._timer.durationInMs(),
  }

  if (['JMS', 'JMS_ENTRY'].indexOf(transaction.type) === -1) dataToSend.callTrace.push(callTrace);
  dataToSend.externalCallDetails = {
    http: [],
    sql: [],
    nosql: [],
    cache: [],
    jms: [],
  }

  for (var i = 0; i < transaction.spans.length; i++) {
    const span = transaction.spans[i];
    span.id = uuid.v4();
    const error = transaction.errors[errorProcessedIndex];

    if (error && error.errorTime < span.startTime) {
      _pushErrorToCallTrace.call(this, error);
      errorProcessedIndex++;
    }

    dataToSend.callTrace.push({
      id: span.id,
      name: span.name,
      type: _getCalltraceType(span),
      duration: span.duration,
      startTime: span.startTime,
    });

    if (span.options && span.options.error) {
      dataToSend.transactionError = true;
      dataToSend.callTrace.push({
        id: span.id,
        name: span.name,
        type: 'EXCEPTION',
        duration: 0,
        startTime: span.startTime,
      });
    }

    if (!span.ended || !span.options || typeof span.options !== 'object') {
      continue;
    }

    const trace = span.options;
    trace.id = span.id;
    trace.resTime = span.duration;
    if (!trace.error) delete trace.error;

    if (span.type === 'http:outbound') {
      dataToSend.externalCallDetails.http.push(trace);
    } else if (appConstants.SQL_DBS.indexOf(span.type) > -1) {
      dataToSend.externalCallDetails.sql.push(trace);
    } else if (appConstants.NOSQL_DBS.indexOf(span.type) > -1) {
      dataToSend.externalCallDetails.nosql.push(trace);
    } else if (appConstants.QUEUES.indexOf(span.type) > -1) {
      // since Java BTM is done like, we need to set all MQ as JMS
      dataToSend.externalCallDetails.jms.push(trace);
    } else {
      logger.debug(appendMsg, "Instrumentation, ignored others type", span.type)
    }
  }

  dataToSend.otherSqlDetails = [];
  _pushToOtherSqlDetails(transaction.fastDBQuery);
  _pushToOtherSqlDetails(transaction.errorExceedDBQuery);
  _pushToOtherSqlDetails(transaction.slowExceedDBQuery);

  if (transaction.errors && transaction.errors.length) {
    dataToSend.transactionError = true;
    dataToSend.errors = transaction.errors;

    for (i = errorProcessedIndex; i < transaction.errors.length; i++) {
      _pushErrorToCallTrace.call(this, transaction.errors[i]);
    }
  }

  dataToSend.externalCallDetails = dataToSend.externalCallDetails;
  dataToSend.processID = process.pid;
  dataToSend.tz = appUtils.getTimezone();
  dataToSend.utcOffset = appUtils.getUtcOffset();
  dataToSend.utc = moment(transaction._timer.start).utc().format('DD/MM/yyyy HH:mm:ss');
  dataToSend.nodeId = this._agent.config.component_id;
  dataToSend.threadId = threads.threadId;

  if (dataToSend.queryString && typeof dataToSend.queryString === 'object') {
    dataToSend.queryString = queryString.stringify(dataToSend.queryString);
  }

  return dataToSend;

  function _pushErrorToCallTrace(error) {
    dataToSend.callTrace.push({
      id: error.id,
      name: this._agent.getErrorOccuredLine(error),
      type: 'EXCEPTION',
      duration: 0,
      startTime: error.errorTime,
    });
  }

  function _pushToOtherSqlDetails(obj) {
    if (!obj) return;
    Object.keys(obj).forEach(key => {
      if (!obj[key]) return;
      dataToSend.otherSqlDetails.push(obj[key]);
    });
  }
}

function _getCalltraceType(span) {
  if (span.type === 'http:outbound') {
    return "HTTP";
  } else if (appConstants.SQL_DBS.indexOf(span.type) > -1) {
    return "SQL";
  } else if (appConstants.QUEUES.indexOf(span.type) > -1) {
    // since Java BTM is done like, we need to set all MQ as JMS
    return span.options.pointcut;
  } else if (appConstants.NOSQL_DBS.indexOf(span.type) > -1 || span.type === 'redis') {
    return span.type.toUpperCase();
  }

  return span.type;
}