'use strict'

const endOfStream = require('end-of-stream');

const logger = require('../logger');
const appUtils = require('../app-utils');
const appConstant = require('../app-constants');
const transaction = require('./transaction-utils');
const resTransformer = require('./http-res-transformer');

const appendMsg = 'Http shared:';
const eGGUID = appConstant.GUID_NAME;

const transactionForResponse = new WeakMap();

exports.instrumentRequest = function (agent, moduleName) {
  const ins = agent._instrumentation;

  return function (orig) {
    return function onRequest(event, req, res) {
      let trans = null;
      try {
        if (event === 'request') {
          logger.debug(appendMsg, 'New incoming', moduleName, 'request call url:', req.url);
          resTransformer.transform(agent, req, res, moduleName);

          //CHECK url is Blacklisted
          if (isUrlExcluded(agent, req.url)) {
            logger.debug(appendMsg, 'ignoring blacklisted request, url:', req.url);
            _ignoreTheRequest();
          } else {
            const option = getRequestIdentifier(req, agent);
            option.method = req.method;
            trans = agent.startTransaction(req.url, `${moduleName}:inbound`, option);

            if (trans) {
              transactionForResponse.set(res, trans);
              ins.bindEmitter(req);
              ins.bindEmitter(res);
              endOfStream(res, _endOfStream); // End of stream
            } else {
              _ignoreTheRequest();
            }
          }
        } else if (event === 'listening' && this.address && typeof this.address === 'function') {
          const address = this.address();
          if (address && address.port) {
            agent.info.serverInfo = agent.info.serverInfo || [];
            agent.info.serverInfo.push({
              scheme: moduleName,
              port: address.port
            });
            logger.debug(appendMsg, moduleName + " is listening to the port " + address.port);
          }
        }
      } catch (e) {
        logger.error(appendMsg, moduleName + " error ", e);
      }

      return orig.apply(this, arguments);

      function _endOfStream(err) {
        try {
          if (trans.ended) return;
          setUriAndQueryString(req, agent.config, trans.requestIdentifier);

          if (!err) {
            endTranaction(trans, res);
            return;
          }

          // Handle case where res.end is called after an error occurred on the
          // stream (e.g. if the underlying socket was prematurely closed)
          const end = res.end;
          res.end = function () {
            const result = end.apply(this, arguments)
            endTranaction(trans, res);
            return result;
          }

          // if error in endOfStream and the trans is not ended even after 60 sec then EG will end the transaction
          setTimeout(() => {
            endTranaction(trans, res);
          }, 60 * 1000);
        } catch (er) {
          deleteTransaction(res);
          logger.error(appendMsg, moduleName + " endOfStream error ", er);
        }
      }

      function _ignoreTheRequest() {
        // don't leak previous transaction
        agent._instrumentation.supersedeWithEmptyRunContext();
        // for testing perpose, we are calling the fn. don't remove this.
        if (agent._externalTransport) agent._externalTransport();
      }
    }
  }
}

function endTranaction(trans, res) {
  deleteTransaction(res);
  if (trans.ended) return;
  trans.requestIdentifier.statusCode = res.statusCode || null;
  trans.requestIdentifier.statusMessage = res.statusMessage || "";

  if (appConstant.REDIRECT_STATUS_CODE.indexOf(res.statusCode) > -1 && res.getHeader('Location')) {
    trans.requestIdentifier.isRequestRedirect = true;
  }

  logger.debug(appendMsg, `call ended with status code: ${res.statusCode}, id: ${trans.id}`);
  trans.end();
}

function getTransaction(res) {
  try {
    return transactionForResponse.get(res);
  } catch (e) { }
}
exports.getTransaction = getTransaction;

function deleteTransaction(res) {
  try {
    transactionForResponse.delete(res);
  } catch (e) { }
}

function setUriAndQueryString(req, config, requestIdentifier) {
  if (req.query && !appUtils.isEmptyObj(req.query)) {
    requestIdentifier.queryString = req.query;
  } else if (req.params && !appUtils.isEmptyObj(req.params)) {
    // for static URL's the query string is like  { '0': 'url' } in express. to avoid this below code
    if ('/' + req.params[0] == req.url || req.params[0] == req.url) {
      delete req.params[0];
    }

    if (!appUtils.isEmptyObj(req.params))
      requestIdentifier.queryString = req.params;
  }

  //For expressjs pattern path
  if (req.route) {
    //To handle '/api/customer/:id' pattern,
    let path = '';

    if (req.route.path) {
      if (typeof req.route.path !== 'string') {
        path = req.route.path.toString();
      } else {
        path = req.route.path;
      }
    } else if (req.route.regexp && req.route.regexp.source) {
      path = req.route.regexp.source;
    }

    if (path && path[path.length - 1] === '?') {
      path = path.substring(0, path.length - 1);
    }

    if (path && path.indexOf('/:') > -1) {
      requestIdentifier.uri = path;
      const uri = req.protocol + '://' + req.get('host');
      if (req.baseUrl && req.baseUrl !== uri && req.baseUrl.substring(0, 4) !== 'http' && req.baseUrl.substring(0, 5) !== 'https') {
        requestIdentifier.uri = req.baseUrl + requestIdentifier.uri
      }
    }
  }

  requestIdentifier.uri = transaction.spliceUrl(requestIdentifier.uri, config.max_url_segments);
}


/**
 * Number of options 
 * Filter by status code
 * filter by Filetype
 * filter by regex pattern 
 * filter urls
 * filter by req header contains req.headers['user-agent'] 
 * @param {} url 
 * @param {*} statusCode 
 */

function isUrlExcluded(agent, url) {
  const splittedURL = url.split('?');
  const urlPath = splittedURL[0];
  const blackListedUrls = agent.config.black_listed_urls;

  try {
    if (blackListedUrls && blackListedUrls instanceof RegExp && blackListedUrls.test(urlPath)) {
      return true;
    }
  } catch (e) {
    logger.error(appendMsg, 'isUrlExcluded fn throw error', e, 'URL:', url)
  }

  return false;
}

exports.isUrlExcluded = isUrlExcluded;

function getRequestIdentifier(req, agent) {
  return transaction.getTransactionWithReqIdentifier(null, agent.config, req.url, req.headers, req.method, req.connection);
}
exports.getRequestIdentifier = getRequestIdentifier;

//OUT BOUND CALLS
exports.traceOutgoingRequest = function (agent, moduleName, method) {
  var ins = agent._instrumentation

  return function (orig) {
    return function () {
      let id = null, span = null;
      try {
        logger.debug(appendMsg, `new ${moduleName}.${method} Oubound call`);
        const reqInfoArg = arguments[0];
        let port = reqInfoArg.port;
        port = port ? ':' + port : '';
        const protocol = moduleName !== 'http' ? 'https' : 'http';
        const uri = protocol + '://' + (reqInfoArg.host || 'localhost') + port + reqInfoArg.path;

        span = agent.startSpan(reqInfoArg.method + ' ' + uri, 'http:outbound', 'Http');
        if (!span) return orig.apply(this, arguments);

        id = span.transaction.id;
        if (id) {
          reqInfoArg.headers = reqInfoArg.headers || {}
          reqInfoArg.headers[eGGUID] = span.getNeweGGUID();
        }

        const options = {};
        options.uri = uri;
        options.nodeOrder = span.nodeOrder;
        options.resTime = 0;
        span.options = options;
      } catch (e) {
        logger.error(appendMsg, 'traceOutgoingRequest', e);
      }

      const req = orig.apply(this, arguments);
      span.options && (span.options.method = req.method);
      ins.bindEmitter(req);
      req.on('response', onresponse);

      //to capture external error
      req.on('error', function (err) {
        if (err) {
          logger.debug(appendMsg, 'error captured at ' + method);
          span.captureError(err);
        }

        span.end();
      });

      return req;

      function onresponse(res) {
        logger.debug(appendMsg, 'Response came for http oubound request', {
          id: id,
          url: span && span.options && span.options.uri
        });
        ins.bindEmitter(res);

        if (res.prependListener) {
          // Added in Node.js 6.0.0
          res.prependListener('end', onEnd);
        } else {
          var existing = res._events && res._events.end;
          if (!existing) {
            res.on('end', onEnd);
          } else {
            if (typeof existing === 'function') {
              res._events.end = [onEnd, existing];
            } else {
              existing.unshift(onEnd);
            }
          }
        }

        function onEnd() {
          const headGuid = appConstant.RES_HEADER_GUID;
          const hostGuid = res.headers ? res.headers[headGuid] || res.headers[headGuid.toLowerCase()] : '';
          if (hostGuid) {
            logger.debug(appendMsg, 'The eG is enabled in the destination server and Guid ' + hostGuid);
            options.tftKey = hostGuid;
          }

          logger.debug(appendMsg, 'http outbound request event is completed id', id);
          span.options.statusCode = res.statusCode || null;
          span.end()
        }
      }
    }
  }
}