'use strict'

const uuid = require('uuid');
const cluster = require('cluster');

const logger = require('../logger');
const appUtils = require('../app-utils');
const pidPort = require('../utils/pid-port');
const ContainerDiscovery = require('./container-discovery');

const appendMsg = 'Discovery:';

function Discovery(agent) {
  this.agent = agent;
  this.isRunning = false;
  this.containerDiscovery = new ContainerDiscovery(agent);
  this.standAlone = !cluster.worker && appUtils.isEmptyObj(cluster.workers);
}

module.exports = Discovery;

Discovery.prototype.start = async function () {
  try {
    const _self = this;
    _reset.call(this);
    this.isDiscoveryDetailsSent = false;
    this.newGuid = `DISCOVER_ME_${uuid.v4()}`;
    const host = this.agent.info.hostName || this.agent.info.ip;
    this.startupDiscovery.POD_IP = host;
    this.startupDiscovery.CONTAINER_IP = host;
    this.startupDiscovery.CONTAINER_HOST_NAME = host;
    this.startupDiscovery.CONTAINER_REQUESTED_IP = host;
    logger.info(appendMsg, 'Stage: 1/4, Discovery has started. Host:', host, logger.doConsole);

    if (this.agent.info.osType.toLowerCase().indexOf('linux') > -1) {
      await this.containerDiscovery.start();
      _setContainerDetails.call(this, host);
    }

    getTempGuid.call(this, 30, (guid1) => {
      if (guid1 && this.standAlone) return _startWatcher.call(this, guid1);

      getTempGuid.call(this, 60, (guid2) => {
        if (guid2 && guid1 === guid2) return _startWatcher.call(this, guid2);

        getTempGuid.call(this, 5 * 60, (guid3) => {
          if (guid3) return _startWatcher.call(this, guid3);
          logger.warn(appendMsg, 'Not able to discover the Guid, so going to use newly generated Guid');
          _self.agent.metadata.set('TEMP_GUID', _self.newGuid);
          _startWatcher.call(this, _self.newGuid);
        });
      });
    });

    this.isRunning = true;
  } catch (e) {
    logger.error(appendMsg, e);
  }
}

function _setContainerDetails(host) {
  if (!this.containerDiscovery.isContainer) return;

  this.startupDiscovery.IS_CONTAINER = true;
  this.startupDiscovery.IS_STANDALONE = false;
  this.startupDiscovery.IS_KUBERNETES = this.containerDiscovery.isPod;
  this.startupDiscovery.POD_IP = this.containerDiscovery.containerIP || this.agent.info.ip || host;
  this.startupDiscovery.CONTAINER_IP = this.containerDiscovery.containerIP || this.agent.info.ip || host;
  this.startupDiscovery.CONTAINER_ID = this.containerDiscovery.containerId || "";
  this.startupDiscovery.CONTAINER_NAME = this.containerDiscovery.containerId || "NONE";
  this.startupDiscovery.POD_NAME = this.containerDiscovery.podName || "NONE";
  this.startupDiscovery.POD_NAME_SPACE = this.containerDiscovery.nameSpace || "NONE";
}

function _reset() {
  this.details = {
    JvmGUID: null,
    appName: 'MyApp',
    tierName: 'MyTier',
    NodeID: null,
    workerId: null,
    SYSTEM_PROPERITES: process.argv,
    SYSTEM_ENVIRONMENT_VARIABLES: process.env,
    CONTAINER_NAME: 'NONE',
  }
  this.startupDiscovery = {
    CONTAINER_PORT: null,
    CONTAINER_IP: "",
    CONTAINER_SERVER_VERSION_NAME: "NODEJS/ " + this.agent.info.nodeVersion,
    POD_IP: "",
    CONTAINER_REQUESTED_PORT: "0000",
    IS_CONTAINER: false,
    IS_KUBERNETES: false,
    IS_STANDALONE: true,
    CONTAINER_ID: "NONE",
    CONTAINER_HOST_NAME: "",
    POD_NAME: "NONE",
    IS_OTHER_TYPES: false,
    POD_NAME_SPACE: "NONE",
    CONTAINER_PID: process.pid,
    CONTAINER_NAME: "NONE",
    CONTAINER_VERSION: "NONE",
    CONTAINER_SCHEME: "http",
    DISCOVERED: "DISCOVERED",
    CONTAINER_SERVER_NAME: "NODEJS",
    CONTAINER_REQUESTED_IP: ""
  }
  this.details.STARTUP_DISCOVERY_DETAILS = this.startupDiscovery;
}

function _startWatcher(guid) {
  const _self = this;
  this.guid = guid;
  this.details.JvmGUID = guid;
  this.details.NodeID = guid;
  logger.info(appendMsg, 'Stage: 2/4, Discovered the temp Guid', guid);

  this.watcher = setInterval(_ => {
    if (!_self.agent.info.serverInfo || !_self.agent.info.serverInfo.length || _self.timeOut) return;
    _self.timeOut = setTimeout(_ => {
      const port = _getPortNo.call(_self);
      _self.startupDiscovery.CONTAINER_PORT = port ? port + '' : null;
      logger.info(appendMsg, 'Port No from HTTP module wrapping', this.agent.info.serverInfo);
      _portsFound.call(_self);
    }, 2 * 1000);
  }, 500);

  this.pidToPortTimer = setTimeout(async function () {
    const isPortFound = await _pidToPorts.call(_self);
    if (!isPortFound && this.agent.info.pm2Details && this.agent.info.pm2Details.isDaemon === true) {
      logger.info(appendMsg, `Stage: 3/4, Since it PM2 Daemon process, it can be allowed even without port no`, this.agent.info.pm2Details);
      _sendDiscoveryDetails.call(this);
    }
  }, 60 * 1000);

  const portno_discovery_timeout = this.agent.getConfig('portno_discovery_timeout') * 1000;
  this.discoveryTimer = setTimeout(() => {
    _finalTry.call(_self, portno_discovery_timeout);
  }, portno_discovery_timeout);
}

function getTempGuid(sec, cb) {
  const _self = this;
  let guid = this.agent.metadata.get('TEMP_GUID');
  if (!guid) {
    _self.agent.metadata.set('TEMP_GUID', _self.newGuid);
  }

  if (this.standAlone) {
    return cb(guid || this.newGuid);
  }

  // It is better to check one more time, to verify
  setTimeout(() => {
    cb(_self.agent.metadata.get('TEMP_GUID'));
  }, sec * 1000);
}

async function _finalTry(sec) {
  const isPortFound = await _pidToPorts.call(this);
  if (isPortFound) return;

  _clearTimers.call(this);
  logger.info(appendMsg, `Even after ${sec} sec, the profiler can't find ports.`);
  this.stop();
}

async function _pidToPorts() {
  let ports = await pidPort.pidToPorts(process.pid);
  ports = Array.from(ports);
  if (!ports || !ports.length) return false;

  _sort(ports);
  this.startupDiscovery.CONTAINER_PORT = ports[0] + '';
  logger.info(appendMsg, `Port No used by pid is`, ports);
  _portsFound.call(this)
  return false;
}

function _clearTimers() {
  if (this.pidToPortTimer) clearTimeout(this.pidToPortTimer);
  if (this.discoveryTimer) clearTimeout(this.discoveryTimer);
  if (this.watcher) clearInterval(this.watcher);
  this.pidToPortTimer = null;
  this.discoveryTimer = null;
  this.watcher = null;
}

function _portsFound() {
  _clearTimers.call(this);
  logger.info(appendMsg, 'Stage: 3/4, Discovered Port number', this.startupDiscovery.CONTAINER_PORT, logger.doConsole);
  _sendDiscoveryDetails.call(this);
}

function _sendDiscoveryDetails() {
  if (this.discoverySender) return;

  const _self = this;
  const interval = this.agent.getConfig('discovery_details_interval') * 1000;
  const reminderInterval = this.agent.getConfig('discovery_reminder_interval') * 1000;

  _self.discoverySender = setInterval(() => {
    if (_self.agent.config.unique_component_id) {
      logger.info(appendMsg, 'Stage: Success, component details is received so going to stop discovery', logger.doConsole);
      _self.stop();
    } else {
      _send.call(_self);
    }
  }, interval);

  _self.discoveryReminderSender = setInterval(() => {
    if (_self.agent.config.unique_component_id) {
      logger.info(appendMsg, 'Stage: Success, component details is received so going to stop discovery', logger.doConsole);
      _self.stop();
    } else {
      if (!_self.isDiscoveryDetailsSent) return;
      _self.agent.collector.add({
        type: 'String',
        message: _self.guid,
        sendImmediately: true,
      });
    }
  }, reminderInterval);

  const config = this.agent.config;
  logger.info(appendMsg, 'Stage: 4/4, Discovery details has been added to the queue', logger.doConsole);
  logger.info(appendMsg, `Agent Host: ${config.agent_host}, Agent port: ${config.agent_port}`, logger.doConsole);
  _send.call(_self);
}

function _send() {
  this.isDiscoveryDetailsSent = this.agent.collector.add({
    discoveryDetails: this.details,
    sendImmediately: true,
  });
}

function _getPortNo() {
  const httpPorts = [];
  const httpsPorts = [];

  for (let i = 0; i < this.agent.info.serverInfo.length; i++) {
    if (this.agent.info.serverInfo[i].scheme === 'https') {
      httpsPorts.push(this.agent.info.serverInfo[i].port)
    } else {
      httpPorts.push(this.agent.info.serverInfo[i].port)
    }
  }

  if (httpsPorts.length === 1) return httpsPorts[0];
  if (httpsPorts.length > 1) {
    if (httpsPorts.indexOf(443) > -1) return 443;
    _sort(httpsPorts);
    return httpsPorts[0];
  }

  if (httpPorts.length === 1) return httpPorts[0];
  if (httpPorts.length > 1) {
    if (httpPorts.indexOf(80) > -1) return 80;
    _sort(httpPorts);
    return httpPorts[0];
  }

  return null;
}

function _sort(arr) {
  arr.sort(function (a, b) {
    return a - b;
  });
}

Discovery.prototype.stop = function () {
  _clearTimers.call(this);
  if (this.discoverySender) clearInterval(this.discoverySender);
  if (this.discoveryReminderSender) clearInterval(this.discoveryReminderSender);
  this.discoveryReminderSender = null;
  this.discoverySender = null;
  this.isRunning = false;
  logger.info(appendMsg, 'stoped...!');
}