'use strict'

const onHeaders = require('on-headers');
const Buffer = require('safe-buffer').Buffer;

const logger = require('../logger');
const parser = require('../parser');
const httpShared = require('./http-shared');
const appConstant = require('../app-constants');

const appendMsg = 'Http-res-transformer:';
const preC = '\n\n  <!-- RUM Header Starts --> \n  ';
const postC = '\n  <!-- RUM Header Ends  --> \n';
const validStatusCode = [200].concat(appConstant.REDIRECT_STATUS_CODE);
const cacheControlNoTransformRegExp = /(?:^|,)\s*?no-transform\s*?(?:,|$)/;

exports.transform = function (agent, req, res, moduleName) {
  let ended = false;
  let isNeedToTrans = false;
  const _write = res.write;
  const rumSnippet = agent.getConfig('rum_snippet');
  const enableRumInjection = agent.getConfig('enable_rum_injection');
  const rumSnippetInjectionPlace = agent.getConfig('rum_snippet_injection_place') || [];

  res.write = function write(chunk, encoding) {
    if (ended) return false;
    if (!this._header) this._implicitHeader();
    if (!isNeedToTrans) return _write.apply(this, arguments);

    logger.debug(appendMsg, `Going to transform ${req.url}, type: ${res.getHeader('Content-Type')}`);
    chunk = resTransformer(chunk);
    return _write.call(this, toBuffer(chunk, encoding), encoding);
  }

  onHeaders(res, function () {
    try {
      const trans = httpShared.getTransaction(this);
      if (!trans) return;

      trans.userContext = parser.getUserContextFromReq(req);
      if (agent.getConfig('enable_rum_btm_transaction_mapping')) {
        const egrumHead = req.headers && (req.headers.EGRUM || req.headers.egrum);
        _setHeader.call(this, 'Server-Timing', `eg-btm-guid;desc="${trans.getGuid()}";dur=${trans.getServerTime()}`);

        if (egrumHead === 'isAjax:true') {
          _setHeader.call(this, appConstant.RUM_HEADER, trans.getDataForRUM());
        } else if (isValidHtmlRes()) {
          const reqCookie = req.headers && (req.headers.cookie || req.headers.Cookie);
          if (reqCookie && reqCookie.indexOf(appConstant.RUM_COOKIE + '=') > -1) {
            _pushCookie.call(this, '; Max-Age=1;');
          }

          let cookie = `${trans.getDataForRUM()}; Max-Age=30; Path=${req.originalUrl || req.url};`;
          moduleName === 'https' && (cookie += 'Secure;');
          _pushCookie.call(this, cookie);
        }
      }

      if (trans.egEnabledInReqOrigin) {
        // need to send unique_component_id in res header
        _setHeader.call(this, appConstant.RES_HEADER_GUID, agent.getConfig('unique_component_id'));
        logger.debug(appendMsg, 'Req is from eG enabled server, so Guid was set in the res header');
      }

      if (rumSnippet && enableRumInjection === true && rumSnippetInjectionPlace.length && isNeedToInjectRumSnippet()) {
        isNeedToTrans = true;
        res.removeHeader('Content-Length');
      }
    } catch (e) {
      logger.error(appendMsg, 'onHeaders', e);
    }

    function _pushCookie(value) {
      const cookie = _getHeader.call(this, 'set-cookie') || [];
      cookie.push(`${appConstant.RUM_COOKIE}=${value} SameSite=Lax`);
      _setHeader.call(this, 'set-cookie', cookie);
    }
  });

  function resTransformer(chunk) {
    try {
      let rumInjectedData = chunk && chunk.toString();
      const tagToInject = getValidTagToInject(rumInjectedData);

      if (tagToInject && tagToInject.tag) {
        let desc = preC + rumSnippet + postC;
        desc = tagToInject.isAfter ? tagToInject.tag + desc : desc + tagToInject.tag;
        chunk = rumInjectedData.replace(tagToInject.tag, desc);
        logger.info(appendMsg, `The RUM snippet is injected in to the response of URL ${req.url}`);
      }

      return chunk;
    } catch (e) {
      logger.error(appendMsg, 'onHeaders', e);
      callback(null, chunk, encoding);
    }
  }

  function getValidTagToInject(html) {
    if (!html) return noTransForm('no body found');
    if (!html.match(/<!doctype html.*?>/gi) && !html.match(/<html.*?>/gi)) {
      // if html tag is not found then ignore it
      return noTransForm('html tag is not found');
    }

    for (let i = 0; i < rumSnippetInjectionPlace.length; i++) {
      const ele = rumSnippetInjectionPlace[i];
      if (!ele || !ele.regex || !ele.type) continue;
      const tags = html.match(ele.regex);
      if (!tags || !tags.length) continue;

      if (ele.type === 'AFTER_LAST') {
        return {
          tag: tags[tags.length - 1],
          isAfter: true,
        };
      } else if (ele.type === 'AFTER') {
        return {
          tag: tags[0],
          isAfter: true,
        };
      } else {
        return { tag: tags[0] };
      }
    }

    return noTransForm('RUM Snippet Injection Place is cant be found');
  }

  function isNeedToInjectRumSnippet() {
    if (!isValidHtmlRes()) return noTransForm('it is not a valid html res');
    if (req.method === 'HEAD') return noTransForm('HEAD request');
    if (validStatusCode.indexOf(res.statusCode) === -1)
      return noTransForm('response status is ' + res.statusCode);
    if (!shouldTransform()) return noTransForm('Cache-Control: no-transform');
    return true;
  }

  function noTransForm(msg) {
    logger.debug(appendMsg, `No need to transform ${req.url} because ${msg}`);
    return false;
  }

  function isValidHtmlRes() {
    var contentType = res.getHeader('Content-Type');
    return contentType && contentType.indexOf('text/html') > -1 && req.url.substr(-4) != '.map';
  }

  function shouldTransform() {
    const cacheControl = res.getHeader('Cache-Control');

    // Don't transform for Cache-Control: no-transform
    // https://tools.ietf.org/html/rfc7234#section-5.2.2.4
    return !cacheControl ||
      !cacheControlNoTransformRegExp.test(cacheControl);
  }
}

function _setHeader(key, value) {
  if (this.setHeader) {
    this.setHeader(key, value);
  } else if (this._headers) {
    this._headers[key] = value;
  }
}

function _getHeader(key) {
  if (this.getHeader) {
    return this.getHeader(key);
  } else if (this._headers) {
    return this._headers[key];
  }
}
exports.getHeader = _getHeader;

function toBuffer(chunk, encoding) {
  return !Buffer.isBuffer(chunk)
    ? Buffer.from(chunk, encoding)
    : chunk;
}