import '../model/log_level.dart';
import '../util/string_utils.dart';

/// Default value for reporting a crash
const bool defaultReportCrash = true;

/// Default value for allowing certificate validation
const bool defaultCertificateValidation = true;

/// Default value for user optin
const bool defaultUserOptIn = false;

/// Default value for log level used within the plugin
const LogLevel defaultLogLevel = LogLevel.Info;

/// Configuration class which can be used for manual instrumentation.
class Configuration {
  final LogLevel _logLevel;
  final bool _reportCrash;
  final String? _beaconUrl;
  final String? _applicationId;
  final bool _certificateValidation;
  final bool _userOptIn;

  /// Constructor of configuration class.
  const Configuration(
      {bool reportCrash = defaultReportCrash,
      LogLevel logLevel = defaultLogLevel,
      String? beaconUrl,
      String? applicationId,
      bool certificateValidation = defaultCertificateValidation,
      bool userOptIn = defaultUserOptIn})
      : _reportCrash = reportCrash,
        _logLevel = logLevel,
        _applicationId = applicationId,
        _beaconUrl = beaconUrl,
        _certificateValidation = certificateValidation,
        _userOptIn = userOptIn;

  /// Returns if crash reporting is enabled.
  bool get isCrashReportingEnabled {
    return _reportCrash;
  }

  /// Check if the properties for the Autostart are available and valid
  /// If [beaconUrl] and [applicationId] are available this will return true
  bool get isAutoStartPropertyAvailable {
    return !StringUtils.isStringNullEmptyOrWhitespace(_beaconUrl) &&
        !StringUtils.isStringNullEmptyOrWhitespace(_applicationId);
  }

  /// Return the current used loglevel
  LogLevel get logLevel => _logLevel;

  /// Generating the Startup configuration for a manual startup. Only
  /// relevant configurations will be passed to the native agents.
  /// Currently this is applicationId, beaconUrl and loglevel.
  Map<String, dynamic> getStartupConfiguration() {
    Map<String, dynamic> startupConfiguration = {};

    if (!StringUtils.isStringNullEmptyOrWhitespace(_applicationId)) {
      startupConfiguration["applicationId"] = _applicationId;
    }

    if (!StringUtils.isStringNullEmptyOrWhitespace(_beaconUrl)) {
      startupConfiguration["beaconUrl"] = _beaconUrl;
    }

    if (_logLevel == LogLevel.Debug) {
      // Only sending the non-default value which is debug
      startupConfiguration["logLevel"] = "debug";
    }

    if (!_certificateValidation) {
      // Only sending the non-default value which is false
      startupConfiguration["certificateValidation"] = false;
    }

    if (_userOptIn) {
      // Only sending the non-default value which is true
      startupConfiguration["userOptIn"] = true;
    }

    if (!_reportCrash) {
      // Only sending the non-default value which is false
      startupConfiguration["crashReporting"] = false;
    }

    return startupConfiguration;
  }
}
