'use strict'

const uuid = require('uuid');

const logger = require('../logger');
const parser = require('../parser');
const symbols = require('../symbols');
const appUtils = require('../app-utils');
const httpShared = require('./http-shared');

const appendMsg = 'error-handler:'
module.exports = errorHanler;

function errorHanler(agent) {
  this._agent = agent;
}

errorHanler.prototype.handle = function (err, option) {
  try {
    let trans = null;
    const agent = this._agent;
    let captureStackTrace = false;

    if (option && option.res) {
      trans = httpShared.getTransaction(option.res);
    } else {
      trans = agent.currTransaction();
    }

    if (trans) {
      if (agent.getConfig('max_exception_trace_limit') < trans.errTranceCnt) {
        logger.debug(appendMsg, 'max_exception_trace_limit is exceeds so ignoring the error', err.message);
        return;
      }

      if (agent.getConfig('exception_fqen_capture_limit') < trans.errFqenCaptureCnt) {
        logger.debug(appendMsg, 'exception_fqen_capture_limit is exceeds so ignoring the error', err.message);
        return;
      }

      if (agent.getConfig('max_exce_stack_trace_capture_limit') > trans.errStackTraceCnt) {
        captureStackTrace = true;
        trans.errStackTraceCnt++;
      }

      trans.errTranceCnt++;
      trans.errFqenCaptureCnt++;
    }

    let parsedErr = this.getProcessableError(err);
    if (!parsedErr || !parsedErr.err) return;

    if (!trans) {
      logger.debug(appendMsg, 'Cant find the current Transaction. so sending infra error.');
      agent.sendInfraError(parsedErr, option);
      return;
    }

    const error = {
      id: uuid.v4(),
      errorTime: Date.now(),
      message: parsedErr.message,
      codeError: parsedErr.causes,
    };
    if (captureStackTrace) error.stack = parsedErr.stack.toString();

    trans.errors.push(error);
    parsedErr.err[symbols.errorReportedSymbol] = true;
    logger.debug(appendMsg, 'Error is pushed into the current Transaction. Id:', trans.id);
  } catch (e) {
    logger.error(appendMsg, 'error in handle fn: ', e);
  }
}

errorHanler.prototype.getProcessableError = function (err) {
  const agent = this._agent;
  let parsedErr = null;

  try {
    const maxCauseDepth = agent.getConfig('max_exception_cause_depth');
    const captureCause = agent.getConfig('capture_cause_of_exception');
    parsedErr = parser.parseError(err, maxCauseDepth, captureCause);

    if (!parsedErr) {
      logger.debug(appendMsg, 'Not able to parse the error ', err);
      return;
    }

    const ignored = agent.getConfig('ignored_exceptions');
    const included = agent.getConfig('included_exceptions');
    const traceLineCount = agent.getConfig('max_exception_trace_line_count');

    if (ignored.length && parsedErr.type && ignored.indexOf(parsedErr.type) > -1) {
      parsedErr.err[symbols.errorReportedSymbol] = true;
      logger.debug(appendMsg, 'ignoring the blacklisted error:', parsedErr.type);
      return;
    }

    if (included.length && included.indexOf(parsedErr.type) === -1) {
      parsedErr.err[symbols.errorReportedSymbol] = true;
      logger.debug(appendMsg, 'ignoring the non-included error:', parsedErr.type);
      return;
    }

    appUtils.spliceStackTrace(parsedErr, traceLineCount);

    if (parsedErr.err[symbols.errorReportedSymbol] === true) {
      logger.debug(appendMsg, 'Error is already processed ', parsedErr.message);
      return;
    }

  } catch (e) {
    logger.error(appendMsg, 'error in getProcessableError fn: ', e);
  }

  return parsedErr;
}

errorHanler.prototype.getErrorOccuredLine = function (err, isLineMatch) {
  if (!err || !err.stack) return '';

  const lines = err.stack.split('\n').slice(1);
  for (let i = 0; i < lines.length; i++) {
    const line = lines[i];
    if (line.match(/^\s*[-]{4,}$/)) continue;
    const lineMatch = line.match(/at (?:(.+)\s+\()?(?:(.+?):(\d+)(?::(\d+))?|([^)]+))\)?/);
    if (lineMatch) {
      return isLineMatch ? lineMatch : line.trim();
    }
  }
};

errorHanler.prototype.getErrorOccuredLineDatils = function (err) {
  const lineMatch = this.getErrorOccuredLine(err, true);
  if (!lineMatch) return;

  let object = null;
  let method = null;
  let functionName = null;
  let typeName = null;
  let methodName = null;
  let isNative = (lineMatch[5] === 'native');

  if (lineMatch[1]) {
    functionName = lineMatch[1];
    let methodStart = functionName.lastIndexOf('.');
    if (functionName[methodStart - 1] == '.')
      methodStart--;
    if (methodStart > 0) {
      object = functionName.substr(0, methodStart);
      method = functionName.substr(methodStart + 1);
      let objectEnd = object.indexOf('.Module');
      if (objectEnd > 0) {
        functionName = functionName.substr(objectEnd + 1);
        object = object.substr(0, objectEnd);
      }
    }
    typeName = null;
  }

  if (method) {
    typeName = object;
    methodName = method;
  }

  if (method === '<anonymous>') {
    methodName = null;
    functionName = null;
  }

  return {
    fileName: lineMatch[2] || null,
    lineNumber: parseInt(lineMatch[3], 10) || null,
    functionName,
    typeName,
    methodName,
    columnNumber: parseInt(lineMatch[4], 10) || null,
    native: isNative,
  };
}