#Requires -Version 3.0
#Requires -Module dnsserver
#This File is in Unicode format.  Do not edit in an ASCII editor.

#region help text

<#
.SYNOPSIS
	Provides DNS server information in the customized format.
.DESCRIPTION
	
	To run the script from a workstation, RSAT is required.
	
	Remote Server Administration Tools for Windows 7 with Service Pack 1 (SP1)
		http://www.microsoft.com/en-us/download/details.aspx?id=7887
		
	Remote Server Administration Tools for Windows 8 
		http://www.microsoft.com/en-us/download/details.aspx?id=28972
		
	Remote Server Administration Tools for Windows 8.1 
		http://www.microsoft.com/en-us/download/details.aspx?id=39296
		
	Remote Server Administration Tools for Windows 10
		http://www.microsoft.com/en-us/download/details.aspx?id=45520

.PARAMETER ComputerName
	Specifies a computer to use to run the script against.
	ComputerName can be entered as the NetBIOS name, FQDN, localhost or IP Address.
	If entered as localhost, the actual computer name is determined and used.
	If entered as an IP address, an attempt is made to determine and use the actual 
	computer name.
	Default is localhost.

.INPUTS
	None.  You cannot pipe objects to this script.
.OUTPUTS
	No objects are output from this script.  
#>
#endregion

#region script parameters
[CmdletBinding(SupportsShouldProcess = $False, ConfirmImpact = "None") ]

Param(
	
	[parameter(Mandatory=$False)] 
	[string]$ComputerName="LocalHost"
	)
#endregion

#region initialize variables
$SaveEAPreference = $ErrorActionPreference
$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False
$scriptStartIdentifier = "*** Start - DNS config data ***"
$scriptEndIdentifier = "*** End - DNS config data ***"
$delimiter = "@@@"
#endregion

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#region script setup function
Function TestComputerName
{
	Param([string]$Cname)
	If(![String]::IsNullOrEmpty($CName)) 
	{
		#get computer name
		#first test to make sure the computer is reachable
		If(Test-Connection -ComputerName $CName -quiet)
		{
			Write-Verbose "$(Get-Date): Server $CName is online."
		}
		Else
		{
			$ErrorActionPreference = $SaveEAPreference
			Write-Error "`n`n`t`tComputer $CName is offline.`nScript cannot continue.`n`n"
			Exit
		}
	}

	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
	}

	#if computer name is an IP address, get host name from DNS
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip).AddressList.IPAddressToString
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
		}
		Else
		{
			Write-Warning "Unable to resolve $CName to a hostname"
		}
	}
	Else
	{
		#computer is online but for some reason $ComputerName cannot be converted to a System.Net.IpAddress
	}

	$Results = Get-DNSServer -ComputerName $CName -EA 0 3>$Null
	If($Null -ne $Results)
	{
		#the computer is a dns server
		$Script:DNSServerData = $Results
		Return $CName
	}
	ElseIf($Null -eq $Results)
	{
		#the computer is not a dns server
		$ErrorActionPreference = $SaveEAPreference
		Write-Error "`n`n`t`tComputer $CName is not a DNS Server.`n`n`t`tRerun the script using -ComputerName with a valid DNS server name.`n`n`t`tScript cannot continue.`n`n"
		Exit
	}
	Return $CName
}

Function ProcessScriptStart
{
	$ComputerName = TestComputerName $ComputerName
}
#end region

#region Process Conditional Forwarders Master server
Function ProcessConditionalForwardersMasterServer
{
	Write-Verbose "$(Get-Date): Processing Conditional Forwarders"
	$DNSZones = $Script:DNSServerData.ServerZone | Where-Object {$_.ZoneType -eq "Forwarder"}
	
	If($? -and $Null -ne $DNSZones)
	{
		ForEach($DNSZone in $DNSZones)
		{
			OutputConditionalForwarder $DNSZone
		}
	}
	ElseIf($? -and $Null -ne $DNSZones)
	{
		Write-Host "There is no Conditional Forwarders data"
	}
	Else
	{
		Write-Host "Conditional Forwarders data could not be retrieved"
	}
}

Function OutputConditionalForwarder
{
	Param([object] $DNSZone)
	Write-Verbose "$(Get-Date): `tProcessing $($DNSZone.ZoneName)"
	#General tab
	Write-Verbose "$(Get-Date): `t`tGeneral"
	Switch ($DNSZone.ReplicationScope)
	{
		"Forest" {$Replication = "All DNS servers in this forest"; break}
		"Domain" {$Replication = "All DNS servers in this domain"; break}
		"Legacy" {$Replication = "All domain controllers in this domain (for Windows 2000 compatibility"; break}
		"None" {$Replication = "Not an Active-Directory-Integrated zone"; break}
		Default {$Replication = "Unknown: $($DNSZone.ReplicationScope)"; break}
	}
	
	$IPAddresses = $DNSZone.MasterServers
	ForEach($ip in $IPAddresses)
	{
		$Resolved = ResolveIPtoFQDN $IP.IPAddressToString
		Write-Host "server name :$($env:ComputerName) $($delimiter) Conditional Forwarder ZoneName : $($DNSZone.ZoneName) $($delimiter) Server FQDN: $($Resolved) $($delimiter) IP Address : $($ip.IPAddressToString) " 
	}
}
#endregion

#region reusable common functions
Function ResolveIPtoFQDN
{
	Param([string]$cname)
	
	try
	{
		$ip = $CName -as [System.Net.IpAddress]
		
		If($ip)
		{
			$Result = [System.Net.Dns]::gethostentry($ip)
			
			If($? -and $Null -ne $Result)
			{
				$CName = $Result.HostName
			}
			Else
			{
				$CName = 'Unable to resolve'
			}
		}
	}
	catch 
	{
		Write-Verbose "Catch Block Executed : ResolveIPtoFQDN"
		$CName = 'Unable to resolve'
	}
	Return $CName
}
#endregion

#region script core
#Script begins
Write-Host $scriptStartIdentifier 
ProcessScriptStart
try
{
	ProcessConditionalForwardersMasterServer
}
catch 
{
	$global:isErrorOccured = $True
	Write-Verbose "catch block Executed : ProcessConditionalForwardersMasterServer"
}
If(-Not $isErrorOccured)
{
	Write-Host $scriptEndIdentifier
}
#endregion