#requires -Version 3.0
#requires -Module DHCPServer
#This File is in Unicode format.  Do not edit in an ASCII editor.

#region help text
<#
.SYNOPSIS
	To get statistics (Scope Name, No. of addresses in use, No. of free addresses, No. of total addresses, Scope status) of all 
	the IPV4 scopes (scope, super scope, multicast scope) and IPV6 scopes
.DESCRIPTION
	To get statistics (Scope Name, No. of addresses in use, No. of free addresses, No. of total addresses, Scope status) of all 
	the IPV4 scopes (scope, super scope, multicast scope) and IPV6 scopes
	
	Script requires at least PowerShell version 3 but runs best in version 5.
	
	Requires the DHCPServer module.
	Can be run on a DHCP server or on a Windows 8.x or Windows 10 computer with RSAT installed.
		
	Remote Server Administration Tools for Windows 8 
		http://www.microsoft.com/en-us/download/details.aspx?id=28972
		
	Remote Server Administration Tools for Windows 8.1 
		http://www.microsoft.com/en-us/download/details.aspx?id=39296
		
	Remote Server Administration Tools for Windows 10
		http://www.microsoft.com/en-us/download/details.aspx?id=45520
	
	For Windows Server 2003, 2008 and 2008 R2, use the following to export and import the 
	DHCP data:
		Export from the 2003, 2008 or 2008 R2 server:
			netsh dhcp server export C:\DHCPExport.txt all
			
			Copy the C:\DHCPExport.txt file to the 2012+ server.
			
		Import on the 2012+ server:
			netsh dhcp server import c:\DHCPExport.txt all
			
		The script can now be run on the 2012+ DHCP server to document the older DHCP 
		information.

	For Windows Server 2008 and Server 2008 R2, the 2012+ DHCP Server PowerShell cmdlets 
	can be used for the export and import.
		From the 2012+ DHCP server:
			Export-DhcpServer -ComputerName 2008R2Server.domain.tld -Leases -File 
			C:\DHCPExport.xml 
			
			Import-DhcpServer -ComputerName 2012Server.domain.tld -Leases -File 
			C:\DHCPExport.xml -BackupPath C:\dhcp\backup\ 
			
			Note: The c:\dhcp\backup path must exist before the Import-DhcpServer 
			cmdlet is run.
	
	Using netsh is much faster than using the PowerShell export and import cmdlets.
	
	Processing of IPv4 Multicast Scopes is only available with Server 2012 R2 DHCP.
	
.PARAMETER ComputerName
	DHCP server to run the script against.
	The computername is used for the report title.
	ComputerName can be entered as the NetBIOS name, FQDN, localhost or IP Address.
	If entered as localhost, the actual computer name is determined and used.
	If entered as an IP address, an attempt is made to determine and use the actual 
	computer name.
	
	If both ComputerName and AllDHCPServers are used, AllDHCPServers is used.
.PARAMETER AllDHCPServers
	The script will process all Authorized DHCP servers that are online.
	"All DHCP Servers" is used for the report title.
	This parameter is disabled by default.
	
	If both ComputerName and AllDHCPServers are used, AllDHCPServers is used.
	This parameter has an alias of ALL.

.INPUTS
	None.  You cannot pipe objects to this script.
.OUTPUTS
	No objects are output from this script.  Customized text output with delimeter.
#>
#endregion

#region initialize variables
$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False

$delimiter = "@@@"
#endregion

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#region DHCP script functions
Function ProcessIPv4SuperScopeStatistics
{
	Write-Verbose "$(Get-Date): Getting IPv4 Superscopes"
	$IPv4Superscopes = Get-DHCPServerV4Superscope -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $IPv4Superscopes)
	{
		ForEach($IPv4Superscope in $IPv4Superscopes)
		{
			If(![string]::IsNullOrEmpty($IPv4Superscope.SuperscopeName))
			{
				$Statistics = Get-DHCPServerV4SuperscopeStatistics -ComputerName $Script:DHCPServerName -Name $IPv4Superscope.SuperscopeName -EA 0
				
				[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
				$AddressesFree = $Statistics.AddressesFree
				$AddressesInUse = $Statistics.AddressesInUse
				if( $IPv4Superscope.status -eq "Active")
				{
					$IPv4Superscope.status = 1;
				}
				elseif( $IPv4Superscope.status -eq "Inactive")
				{
					$IPv4Superscope.status = 0;
				}
				Write-Host "Scopename : $($IPv4Superscope.SuperscopeName) $($delimiter) AddressesInUse : $($AddressesInUse) $($delimiter) AddressesFree : $($AddressesFree)  $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter) state : $($IPv4Superscope.status)"
			}
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Verbose "Error retrieving IPv4 Superscopes"
	}
	
	$IPv4Superscopes = $Null
	[gc]::collect() 
}

Function ProcessIPv4ScopeStatistics
{
	$IPv4Scopes = Get-DHCPServerV4Scope -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $IPv4Scopes)
	{
		ForEach($IPv4Scope in $IPv4Scopes)
		{
			$ScopeIPScopeName = "[$($IPv4Scope.ScopeId)] $($IPv4Scope.Name)"
			$Ipv4ScopeState = $IPv4Scope.state
		   
			$Statistics   =  Get-DhcpServerv4ScopeStatistics -ComputerName $Script:DHCPServerName -ScopeId $IPv4Scope.ScopeId -EA 0
			
			If($? -and $Null -ne $Statistics)
			{
				[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
				$AddressesFree = $Statistics.AddressesFree
				$AddressesInUse = $Statistics.AddressesInUse
				if( $Ipv4ScopeState -eq "Active")
				{
					$Ipv4ScopeState = 1;
				}
				elseif( $Ipv4ScopeState -eq "Inactive")
				{
					$Ipv4ScopeState = 0;
				}
				Write-Host "Scopename: $($ScopeIPScopeName) $($delimiter) AddressesInUse : $($AddressesInUse) $($delimiter) AddressesFree : $($AddressesFree)  $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter) state : $($Ipv4ScopeState)"
			}
			ElseIf(!$?)
			{
				$global:isErrorOccured = $True
				Write-Verbose "Error retrieving scope statistics : $ScopeIPScopeName"
			}					
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Verbose "Error retrieving IPv4 scopes "
	}
	
	$IPv4Scopes = $Null
	[gc]::collect() 
}
Function ProcessIPv4MulticastScopesStatistics
{
	$CmdletName = "Get-DHCPServerV4MulticastScope"
	If(Get-Command $CmdletName -Module "DHCPServer" -EA 0)
	{
		$IPv4MulticastScopes = Get-DHCPServerV4MulticastScope -ComputerName $Script:DHCPServerName -EA 0

		If($? -and $Null -ne $IPv4MulticastScopes)
		{
			ForEach($IPv4MulticastScope in $IPv4MulticastScopes)
			{	
				$multicastScopeState = $IPv4MulticastScope.state	
				$Statistics = Get-DHCPServerV4MulticastScopeStatistics -ComputerName $Script:DHCPServerName -Name $IPv4MulticastScope.Name -EA 0
				
				If($? -and $Null -ne $Statistics)
				{
					[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
					$AddressesFree = $Statistics.AddressesFree
					$AddressesInUse = $Statistics.AddressesInUse
					if( $multicastScopeState -eq "Active")
					{
						$multicastScopeState = 1;
					}
					elseif( $multicastScopeState -eq "Inactive")
					{
						$multicastScopeState = 0;
					}
					Write-Host "Scopename : $($IPv4MulticastScope.Name) $($delimiter) AddressesInUse : $($AddressesInUse) $($delimiter) AddressesFree : $($AddressesFree)  $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter) state : $($multicastScopeState)"
				}
				ElseIf(!$?)
				{
					$global:isErrorOccured = $True
					Write-Verbose "Error retrieving scope statistics"
				}	
			}
		}
		ElseIf(!$?)
		{
			$global:isErrorOccured = $True
			Write-Verbose "Error retrieving IPv4 Multicast scopes"
		}
		
		$IPv4MulticastScopes = $Null
	}
	[gc]::collect() 
}
Function ProcessIPv6ScopesStatistics
{	
	$IPv6Scopes = Get-DHCPServerV6Scope -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $IPv6Scopes)
	{
		ForEach($IPv6Scope in $IPv6Scopes)
		{
			$scopename  = "[$($IPv6Scope.Prefix)] $($IPv6Scope.Name)"
			$Ipv6State = $IPv6Scope.state
			$Statistics = Get-DHCPServerV6ScopeStatistics -ComputerName $Script:DHCPServerName -Prefix $IPv6Scope.Prefix -EA 0
			
			If($? -and $Null -ne $Statistics)
			{		
				[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
				$AddressesFree = $Statistics.AddressesFree
				$AddressesInUse = $Statistics.AddressesInUse
				if( $Ipv6State -eq "Active")
				{
					$Ipv6State = 1;
				}
				elseif( $Ipv6State -eq "Inactive")
				{
					$Ipv6State = 0;
				}
				
				Write-Host "Scopename : $($scopename ) $($delimiter) AddressesInUse : $($AddressesInUse) $($delimiter) AddressesFree : $($AddressesFree)  $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter) state : $($Ipv6State)"
			}
			ElseIf(!$?)
			{
				$global:isErrorOccured = $True
				Write-Verbose "Error retrieving IPv6 scope statistics"
			}
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Verbose "Error retrieving IPv6 scopes"
	}
	$IPv6Scopes = $Null
	[gc]::collect() 
}
#endregion

#region script core
#Script begins
	$Script:DHCPServerName = "LocalHost"	
	ProcessIPv4SuperScopeStatistics
	ProcessIPv4ScopeStatistics
	ProcessIPv4MulticastScopesStatistics
	ProcessIPv6ScopesStatistics
#endregion
# SIG # Begin signature block
# MIIbjAYJKoZIhvcNAQcCoIIbfTCCG3kCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUi527FXOe6zEB1z6/zZN5KyMr
# QLegghWiMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHOFADw3TANBgkqhkiG9w0B
# AQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEwMTAwMDAwMFoXDTMxMDEw
# NjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMTCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpja1HXOhFCvQp1dU2UtAxQ
# tSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9NmqB+in43Stwhd4CGPN4
# bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaPxqPhLxs6t2HWc+xObTOK
# fF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsOjizVI9r0TXhG4wODMSlK
# XAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7EdMMKbaYK02/xWVLwfoYer
# vnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmEvQECAwEAAaOCAbgwggG0
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkwJwYIKwYBBQUHAgEWG2h0
# dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSMEGDAWgBT0tuEgHf4prtLk
# YaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgtHUQ23eNqerwwcQYDVR0f
# BGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
# ZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFz
# c3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJRFRpbWVzdGFtcGluZ0NB
# LmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7znGucgDo5nRv1CclF0CiNH
# o6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJmTe1ppA/2uHDPYuj1UUp4
# eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3Y18jUmmDgvoaU+2QzI2h
# F3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74wOFcz8QRcucbZEnYIpp1
# FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPWH1ub9y4bTxMd90oNcX6X
# t/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHLKNpbh6aKLzCCBTAwggQY
# oAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEx
# MC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmluZyBD
# QTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPjTsxx/DhGvZ3cH0wsx
# SRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5idQ3Gde2qvCchqXYJawO
# eSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQdLEoJrskacLCUvIUZ4qJ
# RdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+rHCiq85/6XzLkqHlOzEc
# z+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE86xnTrXE94zRICUj6whk
# PlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pzINAPZHM8np+mM6n9Gd8l
# k9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUFBwEBBG0wazAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0Eu
# Y3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3JsNC5kaWdpY2VydC5jb20v
# RGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsME8GA1UdIARI
# MEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
# Y2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQWBBRaxLl7KgqjpepxA8Bg
# +S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzANBgkqhkiG
# 9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwtOhrE7zBh134LYP3DPQ/E
# r4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GOYw4TS63XX0R58zYUBor3
# nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEoAjwukaPAJRHinBRHoXpo
# aK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhCMrI2iiQC/i9yfhzXSUWW
# 6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOBm0/GkxAG/AeB+ova+YJJ
# 92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEwggQZoAMCAQICEAqhJdbW
# Mht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAw
# MFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGln
# aUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCCASIwDQYJKoZI
# hvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6gpnFOVQoV7YjSsQOB0Uz
# URB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvuzKb2Mln+
# X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oiPhisEeTwmQNtO4V8CdPu
# XciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287DxgaqwvB8z9
# 8OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgPhH+fMRTWrdXyZMt7HgXQ
# hBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
# ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1bjAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEFBQcBAQRtMGswJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDovL2NybDQuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDBQBgNV
# HSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cu
# ZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggEB
# AHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
# KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIlHsS6HHssIeLWWywUNUME
# aLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN1
# 1ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHKeZR+WfyMD+NvtQEm
# tmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR7
# 9VYzIi8iNrJLokqV2PWmjlIwggYzMIIFG6ADAgECAhAImnSeuC3D7CfqZerb+DUx
# MA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lD
# ZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcgQ0EwHhcNMjEwNTI1MDAw
# MDAwWhcNMjQwNTI5MjM1OTU5WjBxMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEPMA0GA1UEBxMGSXNlbGluMR0wGwYDVQQKExRlRyBJbm5vdmF0aW9u
# cywgSW5jLjEdMBsGA1UEAxMUZUcgSW5ub3ZhdGlvbnMsIEluYy4wggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDEZQnSmuNt/FyaBkaN3skPE6zuzJIe9pDf
# AZHYpsPdplqB5IT6QbS0qQSxDidrxuc1WJj/MXg9ujRmRi0zMkiTQBlcNMy9ev+Y
# S+YrEJuIkVMKkqSZnixrkFl87uKAi9w1JrTMgjZrmOJAkXLzmP5/0jgSBOj05x1V
# 4g1eGBDbFnpOMQr6luGqZVe+vwMHbolyU5p53QvlWzv4JsbNm4qJ+DEvaLx3kUPO
# SFlJl90K0a0S8Tmkt6xdJZ+nNR128I1GKwg8YHuaD5GWfZe9RnLJpFRG2XHXWlh8
# n8o1gCRQIvbeueHfz+ZrG1K916KhgGPjEvK3umhAEb5vS7MKOBEAN7f+TQ22JXAk
# ijRkSxTIMA2nskgiMx4/Ow4xjN8w/KLxyUDB8FhzZeFK5RxffIp/iSlEWYKVh8rk
# uy0ZBE1s42OmwIFMSBm7AR+R05TSuZozLBkVUscSmalhl4/PszdWE35pXnqRH95+
# jx4PzUTlUNCqYmIZYqr+z2OFuRgfuDa6EvYxdv+5GyFUNI6QVG8ggP+X7IMmiAMG
# lp7fHai/li+NjxmjxEXS1e/JLSt+wO8A1nDuam3MefQGWbDmW6DcXYJUmk1vNvUe
# YAfqTTIUeZZgKhQA1sEHRhH+Tgm+BjZvK20XtxA8nKXtQj0K7gpMCDtE6YaTB0uG
# Zeg8IFyjQQIDAQABo4IBxDCCAcAwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt
# 9mV1DlgwHQYDVR0OBBYEFPnIRJrP2BNJLxoE1rWvfgwwDjZKMA4GA1UdDwEB/wQE
# AwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOg
# MYYvaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5j
# cmwwSwYDVR0gBEQwQjA2BglghkgBhv1sAwEwKTAnBggrBgEFBQcCARYbaHR0cDov
# L3d3dy5kaWdpY2VydC5jb20vQ1BTMAgGBmeBDAEEATCBhAYIKwYBBQUHAQEEeDB2
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wTgYIKwYBBQUH
# MAKGQmh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1
# cmVkSURDb2RlU2lnbmluZ0NBLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEB
# CwUAA4IBAQCUCOdwU+8srLBP/gMQIbtsdu2JWX666Z9GPZisW4KGyCS/U+ZCmB/K
# nELeD4StX3JFXz5ddILA5oone/pEud+mEuETBhzb7RCz82WJvk0DEMa4hzMAh5c5
# zmRFGm8WxDZdI20ztcvW0sObqn9UbsGhQDKpFeww91ZtOQ3/2d0KYZKypzGMba53
# w7pWG170nktocFWTeUhkAy0foAJ2jlBOszj2dshTxAa5Fg1dkpEsBS7oiAPpYkox
# V9ldaXJ8Px1navd6PpDW4A781u13f3tP4FVCsNYw4bz4Z5GInCh6/FshaHRJ6dk5
# SsCr29MlpK8UxK9y2Qmb+jEtRxR0QZLfMYIFVDCCBVACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFD1gmFDZoT5lNmc+
# tirpC64BtrFrMA0GCSqGSIb3DQEBAQUABIICABxceQTjR3DY6AhCaAfyJ7KKnKhj
# S2cOAShtYcnBaFnaS+4qdVWZHWRMX5btPNQtgpiD+FiPlAmcH5rykE0X70E+0VmL
# c1ayuDn3O8VIF8u/Iak31vh3gAA2iTnGrRFQ0gapPYU/xx96PXeLmHQY+oq8XKW0
# CPIVL2MU3n1k+EtnVoddpa3P2bva148L3NpmnvkcsE7mSyqf7ecRspBHJeIjGlFY
# h+I6EilyhAvIZefilCDxFiH/Ds73IUHGC8kwu8XoufQEcq9A9UKCx6EYIk/BF7oN
# l0hTEMOS3ryd7tSJJ7bNXFg6wAycs6b03QZ1KHxWIrNzA6K31Rne9NUfFI1GXjFk
# 2YRG1U2EU4biA22QHiS6lGLT/4MUr9QD1eIySGMt9zWQjoBpDGsYIDSvZXbYiOWU
# 2iMCsIzxK5uMB1Lw0+h7wylDjGSg6NRpvcF9r19VaOKA155AlxCUpIXxvSLL2vK6
# LIxkA4mdVkanFLYi2rPPhTdYp8f3pbqZiLzNs9yPuji1mmDzHOHGCDvJ1DYrhGBb
# q2ryxtfvOSdEzSDdv650k0DQ3GKPOycnYp60ry11SchV/27UXk4EgvpbEGCrgv4k
# gyB8tj4BS6eNKnK7/kzzGb67l1q9Zmc50E+1qlpvnZWtTNmKACr9WWdEQLtZpgA8
# pPPiJkleRs7z0iWBoYICMDCCAiwGCSqGSIb3DQEJBjGCAh0wggIZAgEBMIGGMHIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
# RCBUaW1lc3RhbXBpbmcgQ0ECEA1CSuC+Ooj/YEAhzhQA8N0wDQYJYIZIAWUDBAIB
# BQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0y
# MTA5MTYwNjUxMDhaMC8GCSqGSIb3DQEJBDEiBCBGO3w8WZ+MME6QHbtmVVCxjdt2
# 7w9TBZK7zrGxWN/oojANBgkqhkiG9w0BAQEFAASCAQCPwjsHyi+jkSlTolZt9bzj
# 5rYIakjiUL4P56zuG12mqhFg/uehnIDYQ8t9k9GozwG/xfmtSLNdD0RQkNSzXCJw
# unlmbTmmOS0b7FsxVZGRYOe9+xIpCDpj/YrHBq2hAFyebxVr4cBnFolJQbYDWDsF
# iI3jhWn73VlH8D2H/njGIqYSDd5Fpf9AleTdAcaeiQCJcPktykbv0LNqBFD1b2/+
# yODbkvnO+ywoweOj4ZyywHNEK6fLVwD6ZePv3j+aJIiXN314wnVt2l7CO/c2/+AV
# z/TxpSJPzFNY2PrKpuFKHyf87NgIAI6bJ1goSccVwfBFx8+8ySp+nbHmQxTSrMQt
# SIG # End signature block
