'use strict'

const uuid = require('uuid');

const Timer = require('./timer');
const logger = require('../logger');
const symbols = require('../symbols');
const appConstant = require('../app-constants');
const appendMsg = "Span:";

module.exports = Span

function Span(transaction) {
  this.id = uuid.v4() + '-' + Date.now();
  this.transaction = transaction
  this.started = false
  this.truncated = false
  this.ended = false
  this.name = null
  this.type = null
  this._timer = null
  this._pointCutName = null;
  this._agent = transaction._agent
}

Span.prototype._getInfo = function () {
  return {
    id: this.id,
    name: this.name,
    type: this.type,
    _pointCutName: this._pointCutName,
  }
}

Span.prototype.start = function (name, type, _pointCutName) {
  if (this.started) {
    logger.debug(appendMsg, 'tried to call span.start() on already started span', this._getInfo());
    return
  }

  this.started = true;
  this.name = name || this.name || 'unnamed';
  this.type = type || this.type || 'custom';
  this._pointCutName = _pointCutName || this._pointCutName || null;
  this._timer = new Timer();
  logger.debug(appendMsg, 'transaction started id', this._getInfo());
}


Span.prototype.truncate = function () {
  if (!this.started) {
    logger.debug(appendMsg, 'tried to truncate non-started span - ignoring ', this._getInfo());
    return
  } else if (this.ended) {
    logger.debug(appendMsg, 'tried to truncate already ended span - ignoring', this._getInfo());
    return
  }

  this.truncated = true;
  this.end();
  logger.debug(appendMsg, 'span truncated..!', this._getInfo());
}

Span.prototype.end = function (duration) {
  if (!this.started) {
    logger.debug(appendMsg, 'tried to call span.end() on un-started span ', this._getInfo())
    return
  } else if (this.ended) {
    logger.debug(appendMsg, 'tried to call span.end() on already ended span ', this._getInfo())
    return
  }

  this._timer.end(duration);
  this.ended = true
  logger.debug(appendMsg, 'ended', this._getInfo(), 'Options:', this.options);
  // this._agent._instrumentation.addEndedSpan(this);
  this.transaction._recordEndedSpan(this);
}

Span.prototype.remove = function () {
  this.isNotNeeded = true;
  this.transaction.removeSpan(this);
}

Span.prototype.duration = function () {
  if (!this.ended) {
    logger.debug(appendMsg, 'tried to call span.duration() on un-ended span ', this._getInfo());
    return null
  }

  return this._timer.duration()
}

Span.prototype.offsetTime = function () {
  if (!this.started) {
    logger.debug(appendMsg, 'tried to call span.offsetTime() for un-started span ', this._getInfo());
    return null
  }

  return this._timer.offset(this.transaction._timer)
}

Span.prototype.captureError = function (err) {
  if (!err) return false;

  const agent = this._agent;
  let captureStackTrace = false;
  const trans = this.transaction;

  if (agent.getConfig('max_exception_trace_limit') < trans.errTranceCnt) {
    logger.debug(appendMsg, 'max_exception_trace_limit is exceeds so ignoring the error', err.message);
    return false;
  }

  if (agent.getConfig('exception_fqen_capture_limit') < trans.errFqenCaptureCnt) {
    logger.debug(appendMsg, 'exception_fqen_capture_limit is exceeds so ignoring the error', err.message);
    return false;
  }

  if (agent.getConfig('max_exce_stack_trace_capture_limit') > trans.errStackTraceCnt) {
    captureStackTrace = true;
    trans.errStackTraceCnt++;
  }

  this.transaction.errTranceCnt++;
  this.transaction.errFqenCaptureCnt++;
  logger.debug(appendMsg, 'error is going to record', err, this._getInfo());
  const parsedErr = this._agent.getProcessableError(err);
  if (!parsedErr) return false;

  this.options = this.options || {};
  this.options.error = {
    errorTime: Date.now(),
    message: parsedErr.message,
    codeError: parsedErr.causes,
  };
  if (captureStackTrace) this.options.error.stack = parsedErr.stack.toString();
  this.name = this._agent.getErrorOccuredLine(err);
  err[symbols.errorReportedSymbol] = true;
  return true;
}

Span.prototype.getNeweGGUID = function () {
  this.transaction.nodeOrder = this.transaction.nodeOrder || '1';
  this.nodeOrder = this.transaction.nodeOrder + '.' + (++this.transaction.nodeOrderCounter);
  return this.transaction.id + appConstant.GUID_SEPARATOR + this.nodeOrder;
}