'use strict'

const semver = require('semver');

const shimmer = require('../shimmer');
const logger = require('../../logger');

const appendMsg = 'Mongodb-core:';
const firstSpan = Symbol('first-span');
const CURSOR_FNS_FIRST = ['_find', '_getmore'];
const SERVER_FNS = ['insert', 'update', 'remove', 'auth'];

module.exports = function (mongodb, agent, version, enabled) {
  if (!enabled) return mongodb;
  
  try {
    if (!semver.satisfies(version, '>=1.2.19 <3.1.14')) {
      logger.debug(appendMsg, `version ${version} not supported - aborting...`);
      return mongodb;
    }

    if (mongodb.Server) {
      logger.debug(appendMsg, 'shimming mongodb-core.Server.prototype.command');
      shimmer.wrap(mongodb.Server.prototype, 'command', wrapCommand);
      logger.debug(appendMsg, 'shimming mongodb-core.Server.prototype functions:', SERVER_FNS);
      shimmer.massWrap(mongodb.Server.prototype, SERVER_FNS, wrapQuery);
    }

    if (mongodb.Cursor) {
      logger.debug(appendMsg, 'shimming mongodb-core.Cursor.prototype functions:', CURSOR_FNS_FIRST);
      shimmer.massWrap(mongodb.Cursor.prototype, CURSOR_FNS_FIRST, wrapCursor);
    }
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }

  return mongodb;

  function wrapCommand(orig) {
    return function wrappedFunction(ns, cmd) {
      const trans = agent.currTransaction();
      var id = trans && trans.id
      var span

      logger.debug(appendMsg, 'intercepted call to mongodb-core.Server.prototype.command ', { id: id, ns: ns })

      if (trans && arguments.length > 0) {
        var index = arguments.length - 1
        var cb = arguments[index]
        if (typeof cb === 'function' && (span = agent.startSpan('', 'mongodb', 'mongodb-core'))) {
          var type
          if (cmd.findAndModify) type = 'findAndModify'
          else if (cmd.createIndexes) type = 'createIndexes'
          else if (cmd.ismaster) type = 'ismaster'
          else if (cmd.count) type = 'count'
          else type = 'command'

          arguments[index] = wrappedCallback
          span.name = ns + '.' + type;
        }
      }

      return orig.apply(this, arguments)

      function wrappedCallback() {
        logger.debug(appendMsg, 'intercepted mongodb-core.Server.prototype.command callback, id', id)
        span.end()
        return cb.apply(this, arguments)
      }
    }
  }

  function wrapQuery(orig, name) {
    return function wrappedFunction(ns) {
      const trans = agent.currTransaction();
      var id = trans && trans.id
      var span



      logger.debug(appendMsg, 'intercepted call to mongodb-core.Server.prototype.', name, { id, ns })

      if (trans && arguments.length > 0) {
        var index = arguments.length - 1
        var cb = arguments[index]
        if (typeof cb === 'function' && (span = agent.startSpan(ns + '.' + name, 'mongodb', 'mongodb-core'))) {
          arguments[index] = wrappedCallback
        }
      }

      return orig.apply(this, arguments)

      function wrappedCallback() {
        logger.debug(appendMsg, `intercepted mongodb-core.Server.prototype.${name} callback`, id)
        span.end()
        return cb.apply(this, arguments)
      }
    }
  }

  function wrapCursor(orig, name) {
    return function wrappedFunction() {
      const trans = agent.currTransaction();
      var id = trans && trans.id
      var span
      logger.debug(appendMsg, 'intercepted call to mongodb-core.Cursor.prototype.', name, id)

      if (trans && arguments.length > 0) {
        var cb = arguments[0]
        if (typeof cb === 'function') {
          if (name !== 'next' || !this[firstSpan]) {
            var spanName = `${this.ns}.${this.cmd.find ? 'find' : name}`
            agent.startSpan(spanName, 'mongodb', 'mongodb-core')
          }
          arguments[0] = wrappedCallback;
          if (name === 'next') {
            this[firstSpan] = true
          }
        }
      }

      return orig.apply(this, arguments)

      function wrappedCallback() {
        logger.debug(appendMsg, `intercepted mongodb-core.Cursor.prototype.${name} callback, id: ${id}`);
        span.end()
        return cb.apply(this, arguments)
      }
    }
  }
}
