$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

<#
    .DESCRIPTION
   Machines in Desktop Studio or Desktop Director display a Power State of 'Unknown'.
   This can be caused by changes made on the hypervisor to VM metadata. If the VM's 
   unique ID has changed then the XenDesktop database may be unaware of this UID mismatch.
   This function will verify the UID known to XenDesktop for the VMs and compare against
   the UID provided by the hypervisor. If there's a mismatch the ID will be updated 
   in the database.
#>
Add-pssnapin Citrix.*

function Update-HostedMachineId
{
    Param
    (
        [Parameter(Mandatory=$true, ValueFromPipeline=$true, Position=0)]
        [ValidateNotNull()]
        [ValidateNotNullOrEmpty()]
        [Citrix.Broker.Admin.SDK.Machine] $BrokerMachine,
        [string] $Controller = 'localhost',
        [switch] $Restart
    )

    Begin
    {
        $XDHyp = Get-PSDrive XDHyp -ErrorAction Stop
        switch ($Controller)
        {
             localhost {$DNSName = "$($env:COMPUTERNAME).*"}
             Default   {$DNSName = "$($Controller).*"}
        }
        If (Get-BrokerController -DNSName $DNSName)
        {
            $XDHyp.AdminAddress = $Controller
        }
        else
        {
            Write-Host -Message "Server '$Controller' is not a registered Delivery Controller"
            break
        }
    }
    Process {
        Write-Host "Processing VM $($BrokerMachine.HostedMachineName)"
        $XDHypVM = Get-Item -Path XDHyp:\Connections\$($BrokerMachine.HypervisorConnectionName)\*\*\$($BrokerMachine.HostedMachineName).vm -Verbose:$false
        If($XDHypVM.Id -ne $BrokerMachine.HostedMachineId)
        {
            Write-Host "Correcting HostedMachineId"
            if ($pscmdlet.ShouldProcess($BrokerMachine.HostedMachineName, "Correct [HostedMachineId]"))
            {
                $BrokerMachine | Set-BrokerMachine -HostedMachineId $XDHypVM.Id -PassThru
                If ($PSBoundParameters.Restart)
                {
                    Restart-Computer $BrokerMachine.HostedMachineName
                }
            }
            $Action = 'Updated'
        }
        else
        {
            Write-Host "Property [HostedMachineId] is correct."
            $Action = 'Skipped'
        }
        [PSCustomObject]@{
                   Action = $Action
                   BrokerMachine = $BrokerMachine | Add-Member ScriptMethod -Name ToString -Value {$this.HostedMachineName} -Force -PassThru
        }
    }
}
#localhost/ddc fqdn
if($args.count -eq 0) {$controller = 'localhost'} else {$controller = $args[0]} 
if($args.count -eq 2) {$controller = $args[0] ; $restart = $args[1]}

$unknown = @(Get-BrokerMachine -PowerState Unknown)
if($unknown.count -eq 0)
{
    Write-Host "There are 0 machines found in Unknown power state in the given DDC"
}   

if($restart -like '*yes*')
{
    $unknown | Update-HostedMachineId -Controller $controller -Restart
}
else
{
    $unknown | Update-HostedMachineId -Controller $controller
}

Remove-pssnapin citrix.*

