$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

if($args.count -lt 3)
{
	Write-host "syntax: <InactiveDays> <detailsRequired in boolean> <totalOURequired in boolean> <totalGroupsRequired in boolean> <groupname for membership changes>"
	break
}

$InactiveDays = $args[0]
$detailsRequired = $args[1]
$OURequired = $args[2]
$GroupRequired = $args[3]
$GroupFilter = $args[4] 
try {
	Import-Module ActiveDirectory -ErrorAction Stop
}
catch
{
	Write-Host ("Either you do not have the ActiveDirectory module available on your system, or " +
	"your domain controllers do not have the Active Directory Web Services running.")
	break
}

try {
	Import-Module GroupPolicy -ErrorAction Stop
}
catch
{
	Write-host ("Group policy module is not available on your system.")
	break
}

$SearchBase = $(Get-ADRootDSE | Select-Object -ExpandProperty 'defaultNamingContext')
$domainInfo = Get-ADDomain -Current LocalComputer
$domain = $domainInfo.DistinguishedName

$ouProps = @('DistinguishedName','Name','whenChanged', 'whenCreated', 'CanonicalName')

Write-Host "Organizational Unit"
Write-Host "-------------------"
$ous = Get-ADOrganizationalUnit -SearchBase $SearchBase -Filter * -Properties $ouProps
Write-output "Number of OUs found: $($ous.count)"
if($detailsRequired -eq $true -and $OURequired -eq $true)
{
	if($($ous) -ne $null)
	{
		foreach($ou in $ous)
		{
			Write-host "$($ou.DistinguishedName)#~#$($ou.Name)#~#$($ou.WhenCreated)#~#$($ou.whenChanged)"
		}
	}
}

$emptyOuList = @()
if($($ous) -ne $null)
{
	foreach($ou in $ous)
	{
		$objectList = Get-ADObject -Filter * -SearchBase $ou.DistinguishedName -SearchScope OneLevel |
		Where-Object {$report.DistinguishedName -notcontains $_.DistinguishedName} |
		Select-Object -First 1
		if(-not $objectList)
		{
			$emptyOuList += $ou
		}
	}
}

write-host "Number of empty OUs found: $($emptyOuList.count)"
if($detailsRequired -eq $true)
{
	if($($emptyOuList) -ne $null)
	{
		foreach($ou in $emptyOuList)
		{
			Write-Host "$($ou.DistinguishedName)#~# $($ou.WhenCreated)"
		}
	}
}

Write-Host "Groups"
Write-Host "------"
$groupProps = @('DistinguishedName','Name','whenChanged', 'whenCreated', 'Members','isCriticalSystemObject')
$groups = Get-ADGroup -SearchBase $SearchBase -Filter * -Properties $groupProps
if($($groups) -ne $null)
{
	Write-host "Number of groups found: $($groups.count)"
}
if($detailsRequired -eq $true -and $GroupRequired -eq $true)
{
	if($($groups) -ne $null)
	{
		foreach($group in $groups)
		{
			Write-host "$($group.DistinguishedName)#~#$($group.Name)#~#$($group.WhenCreated)#~#$($group.whenChanged)"
		}
	}
}

$emptyGroupList = @()
if($($groups) -ne $null)
{
	foreach($group in $groups)
	{
		if((-not $group.isCriticalSystemObject) -and ($group.Members.Count -eq 0))
		{ 
			$emptyGroupList += $group
		}
	}
}
write-host "Number of empty groups found: $($emptyGroupList.count)"

if($detailsRequired -eq $true)
{
	if($($emptyGroupList) -ne $null)
	{
		foreach($group in $emptyGroupList)
		{
			#Write-Host "$($group.DistinguishedName)#~# $($group.WhenCreated)"
			Write-host "$($group.DistinguishedName)#~#$($group.Name)#~#$($group.WhenCreated)#~#$($group.whenChanged)"
		}
	}
}

Write-Host "Users"
Write-Host "-----"
$today_object = Get-Date
$unused_conditions_met = {
	## Ensure no built-in AD user objects are removed inadvertantly
	!$_.isCriticalSystemObject -and
	## The account is disabled (account cannot be used)
	(!$_.Enabled -or
	## The password is expired (account cannot be used)
	$_.PasswordExpired -or
	## The account has never been used
	!$_.LastLogonDate -or
	## The account hasn't been used for 60 days
	($_.LastLogonDate.AddDays($InactiveDays) -lt $today_object))
}

$unused_accounts = Get-ADUser -SearchBase $SearchBase -Filter * -Properties passwordexpired,lastlogondate,isCriticalSystemobject | Where-Object $unused_conditions_met

if($($unused_accounts) -ne $null)
{
	Write-Host "Number of Unused user accounts found: $($unused_accounts.count)"
}

if($detailsRequired -eq $true)
{
	if($($unused_accounts) -ne $null)
	{
		foreach($user in $unused_accounts)
		{
			if($user.LastLogonDate -eq $null)
			{
				$LastLoggedOnDaysAgo = '-5'
			}
			else
			{
				$LastLoggedOnDaysAgo = ($today_object - $user.LastLogonDate).Days
			}
			Write-Host "$($user.samAccountName)#~#$($user.Enabled)#~#$($user.PasswordExpired)#~#$LastLoggedOnDaysAgo"
		}
	}
}

Write-Host "GPOs"
Write-Host "----"
$disabledGPOs = @()
$gpos = Get-GPO -All -Domain $domain.DNSRoot
Write-Host "Number of GPOs found: $($gpos.count)"
if($($gpos) -ne $null)
{
	foreach($gpo in $gpos)
	{
		if($gpo.GpoStatus -like '*AllSettingsDisabled')
		{
			$disabledGPOs += $gpo
		}
	}
}

Write-Host "Number of disabled GPOs found: $($disabledGPOs.count)"
if($detailsRequired -eq $true)
{
	if($($disabledGPOs) -ne $null)
	{
		foreach($disabledGpo in $disabledGPOs)
		{
			$DisabledSettingsCategory = ([string]$disabledGpo.GpoStatus).TrimEnd('Disabled') 
			Write-host "$($disabledGpo.DisplayName)#~#$DisabledSettingsCategory#~#$($disabledGpo.Id)#~#" -NoNewline
			write-host "$($disabledGpo.Owner)#~#$($disabledGpo.ModificationTime)"
		}
	}
}

$emptyGPOs = @()
if($($gpos) -ne $null)
{
	foreach($gpo in $gpos)
	{
		if($gpo.Computer.DSVersion -eq 0 -and $gpo.User.DSVersion -eq 0)
		{
			$emptyGpos += $gpo
		}
	}
}

Write-host "Number of empty GPOs found: $($emptyGpos.count)"
if($detailsRequired -eq $true)
{
	if($($emptyGPOs) -ne $null)
	{
		foreach($emptyGpo in $emptyGPOs)
		{
			Write-host "$($emptyGpo.DisplayName)#~#$($disabledGpo.Id)#~#" -NoNewline
			write-host "$($disabledGpo.Owner)#~#$($disabledGpo.CreationTime)"
		}
	}
}

$UnlinkedGpos = @()
$GPOXmlObjs = @()
if($($gpos) -ne $null)
{
	foreach ($gpo in $gpos)
	{
		[xml]$oGpoReport = Get-GPOReport -Guid $gpo.ID -ReportType xml
		if ($($oGpoReport.GPO.LinksTo) -eq $null)
		{
			$UnlinkedGpos += $gpo
		}
		$GPOXmlObjs += $oGpoReport
	}
}

Write-host "Number of unlinked GPOs found: $($UnlinkedGpos.count)"
if($detailsRequired -eq $true)
{
	if($($UnlinkedGpos) -ne $null)
	{
		foreach($UnlinkedGpo in $UnlinkedGpos)
		{
			Write-host "$($UnlinkedGpo.DisplayName)#~#$($disabledGpo.Id)#~#" -NoNewline
			write-host "$($disabledGpo.Owner)#~#$($disabledGpo.CreationTime)"
		}
	}
}

$inactiveGpos = @() 
$inactiveGPos = $UnlinkedGpos + $disabledGPOs
Write-host "Number of inactive GPOs found: $($inactiveGpos.count)"

$aDefaultGpos = @('Default Domain Controllers Policy');
$NoSettingEnabledGpos = @()
if($($GPOXmlObjs) -ne $null)
{
	foreach($sGpo in $GPOXmlObjs)
	{
		$xGpo = ([xml]$sGpo).GPO;
		if ($aDefaultGpos -notcontains $xGpo.Name)
		{
			if (($xGpo.User.Enabled -eq 'true' -and ($($xGPO.User.ExtensionData)) -eq $null) -and `
			($xGpo.Computer.Enabled -eq 'true' -and ($($xGpo.Computer.ExtensionData)) -eq $null))
			{
				$NoSettingEnabledGpos += $xGPO
			}
		}
	}
}

$strCurrentTimeZone = (Get-WmiObject win32_timezone).StandardName
$TZ = [System.TimeZoneInfo]::FindSystemTimeZoneById($strCurrentTimeZone)

Write-Host "Number of GPOs with no settings enabled: $($NoSettingEnabledGpos.count)"
if($detailsRequired -eq $true)
{
	if($($NoSettingEnabledGpos) -ne $null)
	{
		foreach($xGPOObj in $NoSettingEnabledGpos)
		{
			$LocalTime = [System.TimeZoneInfo]::ConvertTimeFromUtc($xGPOObj.CreatedTime, $TZ)
			Write-host "$($xGPOObj.Name)#~#$($LocalTime)"
		}
	}
}

$dc_to_use = $(Get-ADRootDSE | Select-Object -ExpandProperty 'dnsHostName')
If ($dc_to_use -eq $null )
{
	Write-Host "Cannot reach $DomainFQDN."
	break
}

If ($GroupFilter -eq $null -or $GroupFilter -eq "")
{
	$searcher_filter = "(&(objectClass=group)(adminCount=1))" 
}
Else
{
	$searcher_filter = "(&(objectClass=group)(name=$GroupFilter))"
}

$today = Get-Date
$config_nc = [string] ([ADSI]"LDAP://$dc_to_use/RootDSE").configurationNamingContext
$TSL_ADSI = [string] ([ADSI]"LDAP://$dc_to_use/CN=Directory Service,CN=Windows NT,CN=Services,$_config_nc").tombstoneLifetime 
If ( $TSL_ADSI -eq "" )
{
	$threshold = 60
}
Else
{ 
	$threshold = $($_TSL_ADSI)
}
$output_obj = @()
$searcher_root = [ADSI]"LDAP://$dc_to_use"
$searcher_properties = "sAMAccountName","msDS-ReplValueMetaData","distinguishedName"
$searcher = New-Object System.DirectoryServices.DirectorySearcher( $searcher_root , $searcher_filter , $searcher_properties )
$searcher.PageSize = 1000
$searcher_results = $searcher.FindAll()
$searcher_results_count = $searcher_results.Count
if($searcher_results_count -eq 0)
{
	Write-Host "Number of group membership changed: 0"
	break
}
$searcher_results | Sort-Object -Property Path | ForEach-Object `
{
	$current_dn                = [string] $_.Properties.distinguishedname
	$current_samaccountname    = [string] $_.Properties.samaccountname
	$current_replvaluemetadata = $_.Properties."msds-replvaluemetadata"
	$current_replvaluemetadata | ForEach-Object `
	{
		# Store the current value in a XML object
		$current_metadata = $_
		# In case we have an ampersand, we escape it to be able to convert the object to XML
		If ( $_current_metadata -ne $null )
		{
			$current_metadata = $current_metadata.Replace("&","&amp;")
		}
		# Convert into XML
		$current_metadata_xml = [XML] $current_metadata
		# For each item in the XML model we read the child
		$current_metadata_xml | ForEach-Object `
		{
			# Check if the linked value is a member attribute
			If ( $_.DS_REPL_VALUE_META_DATA.pszAttributeName -eq "member" )
			{
				$pszObjectDn   = $_.DS_REPL_VALUE_META_DATA.pszObjectDn
				$dwVersion     = $_.DS_REPL_VALUE_META_DATA.dwVersion
				$ftimeDeleted  = $_.DS_REPL_VALUE_META_DATA.ftimeDeleted
				$ftimeCreated  = $_.DS_REPL_VALUE_META_DATA.ftimeCreated
				#$_ftimeLastOriginatingChange = $_.DS_REPL_VALUE_META_DATA.ftimeLastOriginatingChange #No used yet
				# Check if the member has been added or removed before the threshold
				If ( ([datetime] $ftimeDeleted) -ge $today.AddDays(-$threshold) -or ([datetime] $ftimeCreated) -ge $today.AddDays(-$threshold) )
				{
					# Check if the membership is still valid
					If ( $ftimeDeleted -ne "1601-01-01T00:00:00Z")
					{
						$flag_operation = "removed"
						$datemodified   = $ftimeDeleted        
					}
					Else
					{
						$flag_operation = "added"
						$datemodified   = $ftimeCreated
					}
					$LocalTime = [System.TimeZoneInfo]::ConvertTimeFromUtc($ftimeCreated, $TZ)
					$line = "$($current_dn)#~#$($current_samaccountname)#~#$($pszObjectDn)#~#$($flag_operation)#~#$($LocalTime )#~#$($datemodified)#~#$($dwVersion)"
					$output_obj += $line
				}
			}
		}
	}
}
Write-Host "Number of group membership changed: $($output_obj.count)"
if($detailsRequired -eq $true)
{
	if($($output_obj) -ne $null)
	{
		foreach($obj in $output_obj)
		{
			Write-host "$($obj)"
		}
	}
}




# SIG # Begin signature block
# MIIbjAYJKoZIhvcNAQcCoIIbfTCCG3kCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUCR1G0XvxuLqoJiO20W+nM91u
# +UmgghWiMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHOFADw3TANBgkqhkiG9w0B
# AQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEwMTAwMDAwMFoXDTMxMDEw
# NjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMTCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpja1HXOhFCvQp1dU2UtAxQ
# tSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9NmqB+in43Stwhd4CGPN4
# bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaPxqPhLxs6t2HWc+xObTOK
# fF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsOjizVI9r0TXhG4wODMSlK
# XAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7EdMMKbaYK02/xWVLwfoYer
# vnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmEvQECAwEAAaOCAbgwggG0
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkwJwYIKwYBBQUHAgEWG2h0
# dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSMEGDAWgBT0tuEgHf4prtLk
# YaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgtHUQ23eNqerwwcQYDVR0f
# BGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
# ZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFz
# c3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJRFRpbWVzdGFtcGluZ0NB
# LmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7znGucgDo5nRv1CclF0CiNH
# o6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJmTe1ppA/2uHDPYuj1UUp4
# eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3Y18jUmmDgvoaU+2QzI2h
# F3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74wOFcz8QRcucbZEnYIpp1
# FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPWH1ub9y4bTxMd90oNcX6X
# t/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHLKNpbh6aKLzCCBTAwggQY
# oAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEx
# MC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmluZyBD
# QTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPjTsxx/DhGvZ3cH0wsx
# SRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5idQ3Gde2qvCchqXYJawO
# eSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQdLEoJrskacLCUvIUZ4qJ
# RdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+rHCiq85/6XzLkqHlOzEc
# z+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE86xnTrXE94zRICUj6whk
# PlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pzINAPZHM8np+mM6n9Gd8l
# k9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUFBwEBBG0wazAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0Eu
# Y3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3JsNC5kaWdpY2VydC5jb20v
# RGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsME8GA1UdIARI
# MEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
# Y2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQWBBRaxLl7KgqjpepxA8Bg
# +S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzANBgkqhkiG
# 9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwtOhrE7zBh134LYP3DPQ/E
# r4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GOYw4TS63XX0R58zYUBor3
# nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEoAjwukaPAJRHinBRHoXpo
# aK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhCMrI2iiQC/i9yfhzXSUWW
# 6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOBm0/GkxAG/AeB+ova+YJJ
# 92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEwggQZoAMCAQICEAqhJdbW
# Mht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAw
# MFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGln
# aUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCCASIwDQYJKoZI
# hvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6gpnFOVQoV7YjSsQOB0Uz
# URB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvuzKb2Mln+
# X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oiPhisEeTwmQNtO4V8CdPu
# XciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287DxgaqwvB8z9
# 8OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgPhH+fMRTWrdXyZMt7HgXQ
# hBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
# ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1bjAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEFBQcBAQRtMGswJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDovL2NybDQuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDBQBgNV
# HSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cu
# ZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggEB
# AHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
# KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIlHsS6HHssIeLWWywUNUME
# aLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN1
# 1ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHKeZR+WfyMD+NvtQEm
# tmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR7
# 9VYzIi8iNrJLokqV2PWmjlIwggYzMIIFG6ADAgECAhAImnSeuC3D7CfqZerb+DUx
# MA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lD
# ZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcgQ0EwHhcNMjEwNTI1MDAw
# MDAwWhcNMjQwNTI5MjM1OTU5WjBxMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEPMA0GA1UEBxMGSXNlbGluMR0wGwYDVQQKExRlRyBJbm5vdmF0aW9u
# cywgSW5jLjEdMBsGA1UEAxMUZUcgSW5ub3ZhdGlvbnMsIEluYy4wggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDEZQnSmuNt/FyaBkaN3skPE6zuzJIe9pDf
# AZHYpsPdplqB5IT6QbS0qQSxDidrxuc1WJj/MXg9ujRmRi0zMkiTQBlcNMy9ev+Y
# S+YrEJuIkVMKkqSZnixrkFl87uKAi9w1JrTMgjZrmOJAkXLzmP5/0jgSBOj05x1V
# 4g1eGBDbFnpOMQr6luGqZVe+vwMHbolyU5p53QvlWzv4JsbNm4qJ+DEvaLx3kUPO
# SFlJl90K0a0S8Tmkt6xdJZ+nNR128I1GKwg8YHuaD5GWfZe9RnLJpFRG2XHXWlh8
# n8o1gCRQIvbeueHfz+ZrG1K916KhgGPjEvK3umhAEb5vS7MKOBEAN7f+TQ22JXAk
# ijRkSxTIMA2nskgiMx4/Ow4xjN8w/KLxyUDB8FhzZeFK5RxffIp/iSlEWYKVh8rk
# uy0ZBE1s42OmwIFMSBm7AR+R05TSuZozLBkVUscSmalhl4/PszdWE35pXnqRH95+
# jx4PzUTlUNCqYmIZYqr+z2OFuRgfuDa6EvYxdv+5GyFUNI6QVG8ggP+X7IMmiAMG
# lp7fHai/li+NjxmjxEXS1e/JLSt+wO8A1nDuam3MefQGWbDmW6DcXYJUmk1vNvUe
# YAfqTTIUeZZgKhQA1sEHRhH+Tgm+BjZvK20XtxA8nKXtQj0K7gpMCDtE6YaTB0uG
# Zeg8IFyjQQIDAQABo4IBxDCCAcAwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt
# 9mV1DlgwHQYDVR0OBBYEFPnIRJrP2BNJLxoE1rWvfgwwDjZKMA4GA1UdDwEB/wQE
# AwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOg
# MYYvaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5j
# cmwwSwYDVR0gBEQwQjA2BglghkgBhv1sAwEwKTAnBggrBgEFBQcCARYbaHR0cDov
# L3d3dy5kaWdpY2VydC5jb20vQ1BTMAgGBmeBDAEEATCBhAYIKwYBBQUHAQEEeDB2
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wTgYIKwYBBQUH
# MAKGQmh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1
# cmVkSURDb2RlU2lnbmluZ0NBLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEB
# CwUAA4IBAQCUCOdwU+8srLBP/gMQIbtsdu2JWX666Z9GPZisW4KGyCS/U+ZCmB/K
# nELeD4StX3JFXz5ddILA5oone/pEud+mEuETBhzb7RCz82WJvk0DEMa4hzMAh5c5
# zmRFGm8WxDZdI20ztcvW0sObqn9UbsGhQDKpFeww91ZtOQ3/2d0KYZKypzGMba53
# w7pWG170nktocFWTeUhkAy0foAJ2jlBOszj2dshTxAa5Fg1dkpEsBS7oiAPpYkox
# V9ldaXJ8Px1navd6PpDW4A781u13f3tP4FVCsNYw4bz4Z5GInCh6/FshaHRJ6dk5
# SsCr29MlpK8UxK9y2Qmb+jEtRxR0QZLfMYIFVDCCBVACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFAQ3RH+KEVHk/EE2
# MDiLZAqNEV96MA0GCSqGSIb3DQEBAQUABIICACt3nMznAVccXw6AQPapdnSaJZyy
# OCuy7ZFT++MzjSJO0AwSUOJWLHS9agpDLgOoDe6bCMDPMxpsr4XzXgpBOAbOriIG
# T9saAC/rW/MRf4dKZFHLE5odEgZ9hFOfRXJx7bM/sKk1uugym5in4XIukE08q6vo
# 7JHkMS6XVWE8rTJer/a0xQy6vNwQdt7tNidP26avuBI1+6GvjqkbqfHaSHMGJWDm
# DlSWr1K08aZlYHtMtD+kdIvZWn7l5SABiTH8407mhNgL2X1sHJ66g+3tNexW2wOR
# WXL8EY7zr3pEVMBmocXO4HhDCPvZd22HGNjEO6ULjKVyyHFSaGM5xJTEvF2GgvpY
# 9+0zKf8s1/D6eVSFeBYGLQu3VZQTRuPyvYDff/XdylkRAzZ7kFLmBAgBjK9ukBDc
# lYhyRI0JSt1HR7tOSWHaDH7hLHYDRTUEChwS6CCuHM+qUgQO6F6P1Ity5IGrudrc
# b5tO1tldBSuYSPISwjUPoz4fORHjPrkUdNrzzQIh55q3BnJyZJc+rZ7ayg+9FkcS
# lqeN3/uDA0Es/QW7Iu6HpR3vt7VWgN42loqltelRGosuxkgHrBw+7atK7Nn8g6Nb
# IBXFal0I2qTi/eWsSnw22zxJ/dNaAbVFmYChZM4b7YpwHyCo7qAxOLLSBdl/KP/b
# aEbY2Rz0mh9iIBxIoYICMDCCAiwGCSqGSIb3DQEJBjGCAh0wggIZAgEBMIGGMHIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
# RCBUaW1lc3RhbXBpbmcgQ0ECEA1CSuC+Ooj/YEAhzhQA8N0wDQYJYIZIAWUDBAIB
# BQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0y
# MTA5MDExMjEzNTZaMC8GCSqGSIb3DQEJBDEiBCAjMysj6QjIdL8e7unrdM5n+whk
# mqhGRLVGEfUt3oHZbjANBgkqhkiG9w0BAQEFAASCAQBflKOHS2L0evLUgAAiP7AG
# DFEEfyTsCgvVH8v7lvsfK6cjKZGTvOplr5CrNQpQ3w/2WIgQzV8EfpG15GUKUa9Q
# YHwrLKWpfo2hb6XcVaIlcB4blM1AJvmicNwuAF8eg0NOjkP1ICoCTbE4x6OIIn1/
# 4krJ7K4ovSNXbnNKYyLEFbH5xPV2sUAM8Fxc6r9U45zUufRuOqmo+2oXe25tiOkV
# HLMuXBMycQkKz7R46JjPvUYMq0fF1N5bbtAuLuFzp82LVPOkPq38lQ1b0pNlurrS
# E5BOhPsmuReNfDAgA0Do9ZDtpVu9ww6fgXyWw1bqITBPuITJGqz8GtUKIThHwq5/
# SIG # End signature block
