$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

<#
.DESCRIPTION
	This script is used to get DAG - Database Availability Group information
.NOTES
	File Name      : XchgDAGDetails.ps1
	Created        : 06 Feburary 2015
#>

$Global:ErrorActionPreference="SilentlyContinue"
$dags = @(Get-DatabaseAvailabilityGroup -Status)
[int]$replqueuewarning = 8

foreach ($dag in $dags)
{
	$dbcopyReport = @()		#Database copy health report
	$memberReport = @()		#DAG member server health report
	$databaseSummary = @()		#Database health summary report

	# DAG members found
	$dagmembers = @($dag | Select-Object -ExpandProperty Servers | Sort-Object Name)

	# DAG database found
	$dagdatabases = @(Get-MailboxDatabase -Status | Where-Object {$_.MasterServerOrAvailabilityGroup -eq $dag.Name} | Sort-Object Name)

	foreach ($database in $dagdatabases)
	{
		#Custom object for Database
		$objectHash = @{
		"Database" = $database.Identity
		"Mounted on" = "Unknown"
		"Preference" = $null
		"Total Copies" = $null
		"Healthy Copies" = $null
		"Unhealthy Copies" = $null
		"Healthy Queues" = $null
		"Unhealthy Queues" = $null
		"Lagged Queues" = $null
		"Healthy Indexes" = $null
		"Unhealthy Indexes" = $null
		}

		$databaseObj = New-Object PSObject -Property $objectHash
		$dbcopystatus = @($database | Get-MailboxDatabaseCopyStatus)
		foreach ($dbcopy in $dbcopystatus)
		{
			$objectHash = @{
			"Database Copy" = $dbcopy.Identity
			"Database Name" = $dbcopy.DatabaseName
			"Mailbox Server" = $null
			"Activation Preference" = $null
			"Status" = $null
			"Copy Queue" = $null
			"Replay Queue" = $null
			"Replay Lagged" = $null
			"Truncation Lagged" = $null
			"Content Index" = $null
			}

			$dbcopyObj = New-Object PSObject -Property $objectHash
			$mailboxserver = $dbcopy.MailboxServer
			$pref = ($database | Select-Object -ExpandProperty ActivationPreference | Where-Object {$_.Key -eq $mailboxserver}).Value
			$copystatus = $dbcopy.Status
			$copyqueuelength = $dbcopy.CopyQueueLength
			$replayqueuelength = $dbcopy.ReplayQueueLength
			$contentindexstate = $dbcopy.ContentIndexState

			#Checking whether this is a replay lagged copy
			$replaylagcopies = @($database | Select -ExpandProperty ReplayLagTimes | Where-Object {$_.Value -gt 0})
			if ($($replaylagcopies.count) -gt 0)
			{
				$replaylag = $false
				foreach ($replaylagcopy in $replaylagcopies)
				{
					if ($replaylagcopy.Key -eq $mailboxserver)
					{
						$replaylag = $true
					}
				}

			}
			else
			{
				$replaylag = $false
			}
			#end replay lagged copy

			#Checking for truncation lagged copies
			$truncationlagcopies = @($database | Select -ExpandProperty TruncationLagTimes | Where-Object {$_.Value -gt 0})
			if ($($truncationlagcopies.count) -gt 0)
			{
				$truncatelag = $false
				foreach ($truncationlagcopy in $truncationlagcopies)
				{
					if ($truncationlagcopy.Key -eq $mailboxserver)
					{
						$truncatelag = $true
					}
				}
			}
			else
			{
				$truncatelag = $false
			}

			$dbcopyObj | Add-Member NoteProperty -Name "Mailbox Server" -Value $mailboxserver -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Activation Preference" -Value $pref -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Status" -Value $copystatus -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Copy Queue" -Value $copyqueuelength -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Replay Queue" -Value $replayqueuelength -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Replay Lagged" -Value $replaylag -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Truncation Lagged" -Value $truncatelag -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Content Index" -Value $contentindexstate -Force

			$dbcopyReport += $dbcopyObj
		}

		$copies = @($dbcopyReport | Where-Object { ($_."Database Name" -eq $database) })
		$mountedOn = ($copies | Where-Object { ($_.Status -eq "Mounted") })."Mailbox Server"
		if ($mountedOn)
		{
			$databaseObj | Add-Member NoteProperty -Name "Mounted on" -Value $mountedOn -Force
		}

		$activationPref = ($copies | Where-Object { ($_.Status -eq "Mounted") })."Activation Preference"
		$databaseObj | Add-Member NoteProperty -Name "Preference" -Value $activationPref -Force

		$totalcopies = $copies.count
		$databaseObj | Add-Member NoteProperty -Name "Total Copies" -Value $totalcopies -Force

		$healthycopies = @($copies | Where-Object { (($_.Status -eq "Mounted") -or ($_.Status -eq "Healthy")) }).Count
		$databaseObj | Add-Member NoteProperty -Name "Healthy Copies" -Value $healthycopies -Force

		$unhealthycopies = @($copies | Where-Object { (($_.Status -eq "Failed") -or ($_.Status -eq "ServiceDown") -or ($_.Status -eq "Dismounted") -or ($_.Status -eq "Dismounting") -or ($_.Status -eq "DisconnectedAndHealthy") -or ($_.Status -eq "DisconnectedAndResynchronizing") -or ($_.Status -eq "FailedAndSuspended")) }).Count
		$databaseObj | Add-Member NoteProperty -Name "Unhealthy Copies" -Value $unhealthycopies -Force

		$healthyqueues  = @($copies | Where-Object { (($_."Copy Queue" -lt $replqueuewarning) -and (($_."Replay Queue" -lt $replqueuewarning)) -and ($_."Replay Lagged" -eq $false)) }).Count
		$databaseObj | Add-Member NoteProperty -Name "Healthy Queues" -Value $healthyqueues -Force

		$unhealthyqueues = @($copies | Where-Object { (($_."Copy Queue" -ge $replqueuewarning) -or (($_."Replay Queue" -ge $replqueuewarning) -and ($_."Replay Lagged" -eq $false))) }).Count
		$databaseObj | Add-Member NoteProperty -Name "Unhealthy Queues" -Value $unhealthyqueues -Force

		$laggedqueues = @($copies | Where-Object { ($_."Replay Lagged" -eq $true) -or ($_."Truncation Lagged" -eq $true) }).Count
		$databaseObj | Add-Member NoteProperty -Name "Lagged Queues" -Value $laggedqueues -Force

		$healthyindexes = @($copies | Where-Object { ($_."Content Index" -eq "Healthy") }).Count
		$databaseObj | Add-Member NoteProperty -Name "Healthy Indexes" -Value $healthyindexes -Force

		$unhealthyindexes = @($copies | Where-Object { ($_."Content Index" -ne "Healthy") }).Count
		$databaseObj | Add-Member NoteProperty -Name "Unhealthy Indexes" -Value $unhealthyindexes -Force

		$databaseSummary += $databaseObj
	}

	if ($dagmembers -ne $null)
	{
		foreach ($dagmember in $dagmembers)
		{
			$memberObj = New-Object PSObject
			$memberObj | Add-Member NoteProperty -Name "Server" -Value $($dagmember.Name)

			# Checking replication health for DAG member
			$replicationhealth = $dagmember | Invoke-Command {Test-ReplicationHealth}
			foreach ($healthitem in $replicationhealth)
			{
				$memberObj | Add-Member NoteProperty -Name $($healthitem.Check) -Value $($healthitem.Result)
			}
			$memberReport += $memberObj
		}
	}

	# Database Availability Group Helath Summary

	if ($databaseSummary -ne $null)
	{
		foreach ($line in $databaseSummary)
		{
			$dbName = $line."Database"
			$dbMount = $line."Mounted on"
			$dbPref = $line."Preference"
			$totCopies = $line."Total Copies"
			$healthyCopies = $line."Healthy Copies"
			$unhealthyCopies = $line."Unhealthy Copies"
			$healthQueues = $line."Healthy Queues"
			$unhealthyQueues = $line."Unhealthy Queues"
			$laggedQueues = $line."Lagged Queues"
			$healthyInd = $line."Healthy Indexes"
			$unhealthyUnd = $line."Unhealthy Indexes"
			Write-Host ("DAG Health Summary", $dbName, $dbMount, $dbPref, $totCopies, $healthyCopies, $unhealthyCopies, $healthQueues, $unhealthyQueues, $laggedQueues, $healthyInd, $unhealthyUnd) -Separator ";"
		}
	}

	# Database Availability Group Helath Details

	if ($dbcopyReport -ne $null)
	{
		foreach ($line in $dbcopyReport)
		{
			$dbName = $line."Database Name"
			$dbCopy = $line."Database Copy"
			$mbSrvr = $line."Mailbox Server"
			$actPref = $line."Activation Preference"
			$mbStatus = $line."Status"
			$copyQueue = $line."Copy Queue"
			$replayQueue = $line."Replay Queue"
			$replayLag = $line."Replay Lagged"
			$trunLag = $line."Truncation Lagged"
			$contIndex = $line."Content Index"
			Write-Host ("DAG Health Details", $dbName, $dbCopy, $mbSrvr, $actPref, $mbStatus, $copyQueue, $replayQueue, $replayLag, $trunLag, $contIndex) -Separator ";"
		}
	}

	# Database Availability Group Member Health

	if ($memberReport -ne $null)
	{
		foreach ($line in $memberReport)
		{
			$serverName = $line."Server"
			$cluService = $line."Cluster Service"
			$repService = $line."Replay Service"
			$actManager = $line."Active Manager"
			$rpcListener = $line."Tasks RPC Listener"
			$tcpListener = $line."TCP Listener"
			$dagMemberUp = $line."DAG Members Up"
			$cluNetwork = $line."Cluster Network"
			$qurmGroup = $line."Quorum Group"
			$fileQurm = $line."File Share Quorum"
			$dbSuspend = $line."DB Copy Suspended"
			$dbInitialize = $line."DB Initializing"
			$dbDisconnect = $line."DB Disconnected"
			$dbCopyUp = $line."DB Log Copy Keeping Up"
			$dbReplayUp = $line."DB Log Replay Keeping Up"
			Write-Host ("DAG Member Health", $serverName, $cluService, $repService, $actManager, $rpcListener, $tcpListener, $dagMemberUp, $cluNetwork, $qurmGroup, $fileQurm, $dbSuspend, $dbInitialize, $dbDisconnect, $dbCopyUp, $dbReplayUp) -Separator ";"
		}
	}
}
# SIG # Begin signature block
# MIIbjAYJKoZIhvcNAQcCoIIbfTCCG3kCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUqzxRT8Xrhl1uVx+/GuS/CKmn
# CCqgghWiMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHOFADw3TANBgkqhkiG9w0B
# AQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEwMTAwMDAwMFoXDTMxMDEw
# NjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMTCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpja1HXOhFCvQp1dU2UtAxQ
# tSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9NmqB+in43Stwhd4CGPN4
# bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaPxqPhLxs6t2HWc+xObTOK
# fF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsOjizVI9r0TXhG4wODMSlK
# XAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7EdMMKbaYK02/xWVLwfoYer
# vnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmEvQECAwEAAaOCAbgwggG0
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkwJwYIKwYBBQUHAgEWG2h0
# dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSMEGDAWgBT0tuEgHf4prtLk
# YaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgtHUQ23eNqerwwcQYDVR0f
# BGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
# ZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFz
# c3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJRFRpbWVzdGFtcGluZ0NB
# LmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7znGucgDo5nRv1CclF0CiNH
# o6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJmTe1ppA/2uHDPYuj1UUp4
# eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3Y18jUmmDgvoaU+2QzI2h
# F3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74wOFcz8QRcucbZEnYIpp1
# FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPWH1ub9y4bTxMd90oNcX6X
# t/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHLKNpbh6aKLzCCBTAwggQY
# oAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEx
# MC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmluZyBD
# QTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPjTsxx/DhGvZ3cH0wsx
# SRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5idQ3Gde2qvCchqXYJawO
# eSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQdLEoJrskacLCUvIUZ4qJ
# RdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+rHCiq85/6XzLkqHlOzEc
# z+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE86xnTrXE94zRICUj6whk
# PlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pzINAPZHM8np+mM6n9Gd8l
# k9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUFBwEBBG0wazAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0Eu
# Y3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3JsNC5kaWdpY2VydC5jb20v
# RGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsME8GA1UdIARI
# MEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
# Y2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQWBBRaxLl7KgqjpepxA8Bg
# +S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzANBgkqhkiG
# 9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwtOhrE7zBh134LYP3DPQ/E
# r4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GOYw4TS63XX0R58zYUBor3
# nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEoAjwukaPAJRHinBRHoXpo
# aK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhCMrI2iiQC/i9yfhzXSUWW
# 6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOBm0/GkxAG/AeB+ova+YJJ
# 92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEwggQZoAMCAQICEAqhJdbW
# Mht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAw
# MFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGln
# aUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCCASIwDQYJKoZI
# hvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6gpnFOVQoV7YjSsQOB0Uz
# URB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvuzKb2Mln+
# X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oiPhisEeTwmQNtO4V8CdPu
# XciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287DxgaqwvB8z9
# 8OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgPhH+fMRTWrdXyZMt7HgXQ
# hBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
# ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1bjAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEFBQcBAQRtMGswJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDovL2NybDQuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDBQBgNV
# HSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cu
# ZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggEB
# AHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
# KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIlHsS6HHssIeLWWywUNUME
# aLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN1
# 1ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHKeZR+WfyMD+NvtQEm
# tmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR7
# 9VYzIi8iNrJLokqV2PWmjlIwggYzMIIFG6ADAgECAhAImnSeuC3D7CfqZerb+DUx
# MA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lD
# ZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcgQ0EwHhcNMjEwNTI1MDAw
# MDAwWhcNMjQwNTI5MjM1OTU5WjBxMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEPMA0GA1UEBxMGSXNlbGluMR0wGwYDVQQKExRlRyBJbm5vdmF0aW9u
# cywgSW5jLjEdMBsGA1UEAxMUZUcgSW5ub3ZhdGlvbnMsIEluYy4wggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDEZQnSmuNt/FyaBkaN3skPE6zuzJIe9pDf
# AZHYpsPdplqB5IT6QbS0qQSxDidrxuc1WJj/MXg9ujRmRi0zMkiTQBlcNMy9ev+Y
# S+YrEJuIkVMKkqSZnixrkFl87uKAi9w1JrTMgjZrmOJAkXLzmP5/0jgSBOj05x1V
# 4g1eGBDbFnpOMQr6luGqZVe+vwMHbolyU5p53QvlWzv4JsbNm4qJ+DEvaLx3kUPO
# SFlJl90K0a0S8Tmkt6xdJZ+nNR128I1GKwg8YHuaD5GWfZe9RnLJpFRG2XHXWlh8
# n8o1gCRQIvbeueHfz+ZrG1K916KhgGPjEvK3umhAEb5vS7MKOBEAN7f+TQ22JXAk
# ijRkSxTIMA2nskgiMx4/Ow4xjN8w/KLxyUDB8FhzZeFK5RxffIp/iSlEWYKVh8rk
# uy0ZBE1s42OmwIFMSBm7AR+R05TSuZozLBkVUscSmalhl4/PszdWE35pXnqRH95+
# jx4PzUTlUNCqYmIZYqr+z2OFuRgfuDa6EvYxdv+5GyFUNI6QVG8ggP+X7IMmiAMG
# lp7fHai/li+NjxmjxEXS1e/JLSt+wO8A1nDuam3MefQGWbDmW6DcXYJUmk1vNvUe
# YAfqTTIUeZZgKhQA1sEHRhH+Tgm+BjZvK20XtxA8nKXtQj0K7gpMCDtE6YaTB0uG
# Zeg8IFyjQQIDAQABo4IBxDCCAcAwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt
# 9mV1DlgwHQYDVR0OBBYEFPnIRJrP2BNJLxoE1rWvfgwwDjZKMA4GA1UdDwEB/wQE
# AwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOg
# MYYvaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5j
# cmwwSwYDVR0gBEQwQjA2BglghkgBhv1sAwEwKTAnBggrBgEFBQcCARYbaHR0cDov
# L3d3dy5kaWdpY2VydC5jb20vQ1BTMAgGBmeBDAEEATCBhAYIKwYBBQUHAQEEeDB2
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wTgYIKwYBBQUH
# MAKGQmh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1
# cmVkSURDb2RlU2lnbmluZ0NBLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEB
# CwUAA4IBAQCUCOdwU+8srLBP/gMQIbtsdu2JWX666Z9GPZisW4KGyCS/U+ZCmB/K
# nELeD4StX3JFXz5ddILA5oone/pEud+mEuETBhzb7RCz82WJvk0DEMa4hzMAh5c5
# zmRFGm8WxDZdI20ztcvW0sObqn9UbsGhQDKpFeww91ZtOQ3/2d0KYZKypzGMba53
# w7pWG170nktocFWTeUhkAy0foAJ2jlBOszj2dshTxAa5Fg1dkpEsBS7oiAPpYkox
# V9ldaXJ8Px1navd6PpDW4A781u13f3tP4FVCsNYw4bz4Z5GInCh6/FshaHRJ6dk5
# SsCr29MlpK8UxK9y2Qmb+jEtRxR0QZLfMYIFVDCCBVACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFEa/SzVRDXY470nZ
# rjB+DkzTW20UMA0GCSqGSIb3DQEBAQUABIICADJiQbZt62viJ8VLsu69kBBMCmKb
# CcOtPCh5TACEODqdzi9ylmHRuPkU/EkWTLFJJweiB1IE3CfN99CXy/C96t7JohN/
# bIab456uYQHhgSoPl40tB1ajFTDqa2bOQV895tFtyLKndaTWyYX2K/3GYXkHJRIv
# wDCGHm1yY+evkI0bSyxeikm++m6C41Si2HxTg47+bfAXrDRAjIrFeqVe3BWd93ji
# s5RwEgEdJGoDVuI3+4TqnL0zTSrmQmFZKP9CjFRXJIq0+46d7oeh9MQLQzk8T2mr
# btkJ+sJofV+eg5WZWCPNpWr3rESmY50QZDSSsj3/Omm/wdaTgW0bnVOPFlGjQbiP
# YclWOO0JqFwO+Kd3Icawu1eEHBQHtfSkQG1ikPbTJI41eBSyiskJotS7i4AqoVn7
# 4Tbn7UkPKMrzq8hrHD7/wKtTrYpeB5BLE19sifeUtWEjh1+3a8Die1BIvP+ZcG2U
# NY2XFtJgmutbPYR4BHM8HYlySSBtfVo5XKP2xGXgs11q4xaMQPWb6Xgkc9P/N/5N
# T/Bq+mkyXSzZKQ+5FjcfEzTtZAqwv9IxQW3tZx4XzfnxV4P4z2el32pJeQ4+0mWW
# NGLRfYfYHoOIrB1cx4O6aJ+kKi+PdQq5OmoBLEDC2Ry3O494bI8YyKXJU1VAwiF/
# w5IV59Wry/5/ZYU0oYICMDCCAiwGCSqGSIb3DQEJBjGCAh0wggIZAgEBMIGGMHIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
# RCBUaW1lc3RhbXBpbmcgQ0ECEA1CSuC+Ooj/YEAhzhQA8N0wDQYJYIZIAWUDBAIB
# BQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0y
# MTA5MDExMjM3MDZaMC8GCSqGSIb3DQEJBDEiBCDg+GEyaSNiiJnzqI9+Nxnr5pkq
# exM4IiH/SpjH/kRadzANBgkqhkiG9w0BAQEFAASCAQAnNeM24ziG85mTmxVnYwLk
# JUOVhkAtQNlObZ/1zYUoaR9ZMOWWLc+3JzirH5uS2uRiTWJR4Iy2xAY/eH9fQIfv
# Z/nTSCucUKGUVIikeo12nL2O7cbxA6iD3BI6D1kugptCilCoTU9WF8GcradgmdV/
# kxR6UXIfSb5eWqlloNp+k8cNssNZw4EUH5A1+6JJshXus0gaBZGN+4gCNum15AtB
# GGX0N0RWRDiHp4WxBzrCd9R2ix9lFexE4ZsRVtYyzCnBTF6d6PHGENJMAbCOAsuw
# IwSGZ9Qdod/R0u4ODie7sYVfi9kzR796LZ7Ux75RyyKlPqZ/q1Ex9yejPtJa0L9f
# SIG # End signature block
