# This Script will give all details of Citrix XenDesktop Server OS Machines 
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

$Global:ErrorActionPreference="SilentlyContinue"
Add-PSSnapin "Citrix.Common.Commands"
Add-PSSnapin "Citrix.Host.Admin.V2"
Add-PSSnapin "Citrix.Broker.Admin.V2"

$FQDN = [System.Net.Dns]::GetHostEntry([System.Net.Dns]::GetHostName()).HostName
#Write-Host("FQDN=" + $FQDN);

<#
Function Get-BrokerMachinesPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "MachineName")
	$desktops = @()
	$desktops += Get-BrokerMachine -SessionSupport 'MultiSession' -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($desktops.count / $total * 100, 1)
		#Write-Host  "Getting Broker Machines. Retrieved $($desktops.count) of $total machines completed in $remainingPerc%"
		$last = $desktops[-1].MachineName
		$desktops += Get-BrokerMachine -SessionSupport 'MultiSession' -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { MachineName -gt $last } @argumentList
	}
	return $desktops
}
#>

Function Get-BrokerMachinesPaged
{
	param (
	$DesktopGroupUid,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$desktops = @()
	$LastUid = 0;
	do
	{
		$NewDesktops = @(Get-BrokerMachine -SessionSupport 'MultiSession' -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewDesktops.Count -gt 0)
		{
			$desktops += $NewDesktops
			$LastUid = $desktops[-1].Uid
		}
	} while ($NewDesktops.Count -ge $MaxRecordCount)
	return $desktops
}

#Write-Host "Server OS Machines"
#Write-Host "------------------"
#$BrokerMachines = Get-BrokerMachine -SessionSupport 'MultiSession' -MaxRecordCount 2147483647
$BrokerMachines = Get-BrokerMachinesPaged
if($BrokerMachines -ne $null)
{
	foreach($BrokerMachine in $BrokerMachines)
	{
		$MachineName = $BrokerMachine.MachineName
		if ($MachineName -eq $null -or $MachineName.length -eq 0)
		{
			continue
		}
		$IPAddress = $BrokerMachine.IPAddress
		if ($IPAddress -eq $null -or $IPAddress.length -eq 0)
		{
			$IPAddress = "-"
		}
		Write-Host($BrokerMachine.DesktopGroupName,$MachineName,$IPAddress) -Separator ";"
	}
}


Remove-PSSnapin "Citrix.Common.Commands"
Remove-PSSnapin "Citrix.Host.Admin.V2"
Remove-PSSnapin "Citrix.Broker.Admin.V2"


