'use strict'

const logger = require('../logger');
const appInfo = require('../app-info');
const threads = require('../utils/threads');

let CpuStats = null;
const appendMsg = 'CPU:';

function CPU() {
  if (process.cpuUsage) {
    this.usageExists = true;
  } else {
    this.usageExists = false;
    logger.error(appendMsg, 'Process.cpuUsage is not a function, cpu usage supported from nodejs V6.1.0')
  }
}

module.exports = CPU;

CPU.prototype.start = function () {
  if (!this.usageExists) return;
  this.startTime = process.hrtime();
  this.startUsage = process.cpuUsage();

  if (this.cpuStats) {
    this.cpuStats.reset();
  } else if (appInfo.osType !== 'Windows_NT') {
    try {
      CpuStats = require('cpu-stats');
      this.cpuStats = new CpuStats();
      logger.info(appendMsg, 'CPU Percentage Of Thread wil be reported using cpu-stats module');
    } catch (e) {
      logger.error(appendMsg, "the cpu-stats module couldn't be required, so can't report CPU Percentage Of Thread", e);
    }
  }
  logger.info(appendMsg, `started successfully...!`);
}

CPU.prototype.getUsage = function () {
  if (!this.usageExists) return null;

  try {
    const temp = {};
    if (this.cpuStats) {
      const cpuStats = this.cpuStats.getStats();
      temp.threadCpuPct = cpuStats && cpuStats.threadCpuPct;
    }

    if (!threads.isMainThread) {
      if (!this.cpuStats) return;
      return temp;
    }

    const elapTimeMS = _hrtimeToMS(process.hrtime(this.startTime));
    const cpuUsage = process.cpuUsage(this.startUsage);
    const userUsage = cpuUsage.user / 1000.0;
    const systemUsage = cpuUsage.system / 1000.0;
    this.startTime = process.hrtime();
    this.startUsage = process.cpuUsage();

    temp.userTime = _roundOff(userUsage);
    temp.systemTime = _roundOff(systemUsage);
    temp.userPct = _roundOff(100.0 * userUsage / elapTimeMS);
    temp.systemPct = _roundOff(100.0 * systemUsage / elapTimeMS);

    return temp;
  } catch (e) {
    logger.error(appendMsg, 'getCpuUsage', e)
  }
}

function _roundOff(no) {
  return no.toFixed(2);
}

function _hrtimeToMS(hrtime) {
  return hrtime[0] * 1000.0 + hrtime[1] / 1000000.0;
}

exports.getCpuUsage;