#Requires -Version 3.0
#Requires -Module dnsserver
#This File is in Unicode format.  Do not edit in an ASCII editor.

#region help text

<#
.SYNOPSIS
	Provides DNS server information in the customized format.
.DESCRIPTION
	
	To run the script from a workstation, RSAT is required.
	
	Remote Server Administration Tools for Windows 7 with Service Pack 1 (SP1)
		http://www.microsoft.com/en-us/download/details.aspx?id=7887
		
	Remote Server Administration Tools for Windows 8 
		http://www.microsoft.com/en-us/download/details.aspx?id=28972
		
	Remote Server Administration Tools for Windows 8.1 
		http://www.microsoft.com/en-us/download/details.aspx?id=39296
		
	Remote Server Administration Tools for Windows 10
		http://www.microsoft.com/en-us/download/details.aspx?id=45520

.PARAMETER ComputerName
	Specifies a computer to use to run the script against.
	ComputerName can be entered as the NetBIOS name, FQDN, localhost or IP Address.
	If entered as localhost, the actual computer name is determined and used.
	If entered as an IP address, an attempt is made to determine and use the actual 
	computer name.
	Default is localhost.

.INPUTS
	None.  You cannot pipe objects to this script.
.OUTPUTS
	No objects are output from this script.  
#>
#endregion

#region script parameters
[CmdletBinding(SupportsShouldProcess = $False, ConfirmImpact = "None") ]

Param(
	
	[parameter(Mandatory=$False)] 
	[string]$ComputerName="LocalHost"
	)
#endregion

#region initialize variables
$SaveEAPreference = $ErrorActionPreference
$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False
$scriptStartIdentifier = "*** Start - DNS config data ***"
$scriptEndIdentifier = "*** End - DNS config data ***"
$delimiter = "@@@"
#endregion

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#region script setup function
Function TestComputerName
{
	Param([string]$Cname)
	If(![String]::IsNullOrEmpty($CName)) 
	{
		#get computer name
		#first test to make sure the computer is reachable
		If(Test-Connection -ComputerName $CName -quiet)
		{
			Write-Verbose "$(Get-Date): Server $CName is online."
		}
		Else
		{
			$ErrorActionPreference = $SaveEAPreference
			Write-Error "`n`n`t`tComputer $CName is offline.`nScript cannot continue.`n`n"
			Exit
		}
	}

	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
	}

	#if computer name is an IP address, get host name from DNS
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip).AddressList.IPAddressToString
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
		}
		Else
		{
			Write-Warning "Unable to resolve $CName to a hostname"
		}
	}
	Else
	{
		#computer is online but for some reason $ComputerName cannot be converted to a System.Net.IpAddress
	}

	$Results = Get-DNSServer -ComputerName $CName -EA 0 3>$Null
	If($Null -ne $Results)
	{
		#the computer is a dns server
		$Script:DNSServerData = $Results
		Return $CName
	}
	ElseIf($Null -eq $Results)
	{
		#the computer is not a dns server
		$ErrorActionPreference = $SaveEAPreference
		Write-Error "`n`n`t`tComputer $CName is not a DNS Server.`n`n`t`tRerun the script using -ComputerName with a valid DNS server name.`n`n`t`tScript cannot continue.`n`n"
		Exit
	}
	Return $CName
}

Function ProcessScriptStart
{
	$ComputerName = TestComputerName $ComputerName
}
#endregion

#region Process DNS Server Information
Function ProcessDNSServer
{
	$DNSServerSettings = $Script:DNSServerData.ServerSetting
	$DNSForwarders = $Script:DNSServerData.ServerForwarder
	$DNSServerRecursion = $Script:DNSServerData.ServerRecursion
	$DNSServerCache = $Script:DNSServerData.ServerCache
	$DNSServerScavenging = $Script:DNSServerData.ServerScavenging
	$DNSServerDiagnostics = $Script:DNSServerData.ServerDiagnostics
	
	OutputDNSServer $DNSServerSettings $DNSForwarders  $DNSServerRecursion $DNSServerCache $DNSServerScavenging $DNSServerDiagnostics
}

Function OutputDNSServer
{
	Param([object] $ServerSettings,[object] $DNSForwarders, [object] $ServerRecursion, [object] $ServerCache, [object] $ServerScavenging, [object] $ServerDiagnostics)
	
	try
	{
		$AllIPs = Get-RegistryValue "HKLM:\SYSTEM\CurrentControlSet\Services\DNS\Parameters" "ListenAddresses" $ComputerName	
	}
	catch
	{
		Write-Verbose "catch block while getting all IPs"
	}

	#Interfaces tab
	$listenon = ""
	$listenIpaddress = ""
	$UseRootHints = ""
	$ServerVersion = ""
	$Recursion = ""
	$Bind = ""
	$FailOnLoad = ""
	$RoundRobin = ""
	$NetMask = ""
	$Pollution = ""
	$DNSSEC = ""
	$NameCheck = ""
	$LoadZone = ""
	$EnableScavenging = ""
	$ScavengingIntervalperiod = ""
	$LogLevel = ""
	
	If($Null -eq $AllIPs)
	{
		$listenon += "All IP addresses"
		ForEach($IP in $ServerSettings.ListeningIPAddress)
		{
			$listenIpaddress += " " + $IP 
		}
	}
	Else
	{
		$listenon += "Only the following IP addresses : " 
		ForEach($IP in $ServerSettings.ListeningIPAddress)
		{
			$listenIpaddress += " " + $IP 
		}
	}
	
	#Forwarders tab
	If($DNSForwarders.UseRootHint)
	{
		$UseRootHints = "Yes"
	}
	Else
	{
		$UseRootHints = "No"
	}
	
	#Advanced tab
	$ServerVersion = "$($ServerSettings.MajorVersion).$($ServerSettings.MinorVersion).$($ServerSettings.BuildNumber) (0x{0:X})" -f $ServerSettings.BuildNumber

	If($ServerRecursion.Enable)
	{
		$Recursion = "Not Selected"
	}
	Else
	{
		$Recursion = "Selected"
	}
	
	If($ServerSettings.BindSecondaries)
	{
		$Bind = "Selected"
	}
	Else
	{
		$Bind = "Not Selected"
	}

	If($ServerSettings.StrictFileParsing)
	{
		$FailOnLoad = "Selected"
	}
	Else
	{
		$FailOnLoad = "Not Selected"
	}
	
	If($ServerSettings.RoundRobin)
	{
		$RoundRobin = "Selected"
	}
	Else
	{
		$RoundRobin = "Not Selected"
	}

	If($ServerSettings.LocalNetPriority)
	{
		$NetMask = "Selected"
	}
	Else
	{
		$NetMask = "Not Selected"
	}
	
	If($ServerRecursion.SecureResponse -and $ServerCache.EnablePollutionProtection)
	{
		$Pollution = "Selected"
	}
	Else
	{
		$Pollution = "Not Selected"
	}
	
	If($ServerSettings.EnableDnsSec )
	{
		$DNSSEC = "Selected"
	}
	Else
	{
		$DNSSEC = "Not Selected"
	}
	
	Switch ($ServerSettings.NameCheckFlag)
	{
		0 {$NameCheck = "Strict RFC (ANSI)"; break}
		1 {$NameCheck = "Non RFC (ANSI)"; break}
		2 {$NameCheck = "Multibyte (UTF8)"; break}
		3 {$NameCheck = "All names"; break}
		Default {$NameCheck = "Unknown: NameCheckFlag Value is $($ServerSettings.NameCheckFlag)"}
	}
	
	Switch ($ServerSettings.BootMethod)
	{
		3 {$LoadZone = "From Active Directory and registry"; break}
		2 {$LoadZone = "From registry"; break}
		Default {$LoadZone = "Unknown: BootMethod Value is $($ServerSettings.BootMethod)"; break}
	}
	
	If($ServerScavenging.ScavengingInterval.days -gt 0 -or  $ServerScavenging.ScavengingInterval.hours -gt 0)
	{
		$EnableScavenging = "Selected"
		If($ServerScavenging.ScavengingInterval.days -gt 0)
		{
			$ScavengingInterval = "$($ServerScavenging.ScavengingInterval.days) days"
		}
		ElseIf($ServerScavenging.ScavengingInterval.hours -gt 0)
		{
			$ScavengingInterval = "$($ServerScavenging.ScavengingInterval.hours) hours"
		}
	}
	Else
	{
		$EnableScavenging = "Not Selected"
	}
	
	
	If($EnableScavenging -eq "Selected")
	{
		$ScavengingIntervalperiod =  $ScavengingInterval
	}
	
	#Event Logging
	Switch ($ServerDiagnostics.EventLogLevel)
	{
		0 {$LogLevel = "No events"; break}
		1 {$LogLevel = "Errors only"; break}
		2 {$LogLevel = "Errors and warnings"; break}
		4 {$LogLevel = "All events"; break}	#my value is 7, everyone else appears to be 4
		7 {$LogLevel = "All events"; break}	#leaving as separate stmts for now just in case
		Default {$LogLevel = "Unknown: EventLogLevel Value is $($ServerDiagnostics.EventLogLevel)"; break}
	}
	
	Write-Host "server name : $($env:ComputerName) $($delimiter) Interfaces Listen on : $($listenon) $($delimiter) ipaddress interfaces listen on : $($listenIpaddress) $($delimiter) Forwarders Number of seconds before forward queries time out : $($DNSForwarders.Timeout) $($delimiter) Forwarders Use root hint if no forwarders are available : $($UseRootHints)  $($delimiter) Server version number : $($ServerVersion) $($delimiter) Disable recursion (also disables forwarders) :$($Recursion) $($delimiter) Enable BIND secondaries :  $($Bind) $($delimiter) Fail on load if bad zone data :  $($FailOnLoad)  $($delimiter) Enable round robin : $($RoundRobin)  $($delimiter) Enable netmask ordering : $($NetMask) $($delimiter) Secure cache against pollution : $($Pollution) $($delimiter) 	Enable DNSSec validation for remote responses : $($DNSSEC) $($delimiter) Name checking : $($NameCheck) $($delimiter) Load zone data on startup : $($LoadZone) $($delimiter)	 Enable automatic scavenging of stale records :  $($EnableScavenging)  $($delimiter) Scavenging period : $($ScavengingIntervalperiod)  $($delimiter) Log the following events : $($LogLevel)"
}
#endregion

#region script core
#Script begins
Write-Host $scriptStartIdentifier 
ProcessScriptStart
try
{	
	ProcessDNSServer
}
catch 
{
	$global:isErrorOccured = $True
	Write-Verbose "catch block Executed : OutputDNSServer"
}

If(-Not $isErrorOccured)
{
	Write-Host $scriptEndIdentifier
}
#endregion