$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

if($args.count -lt 3)
{
	Write-host "syntax: <InactiveDays> <detailsRequired in boolean> <totalOURequired in boolean> <totalGroupsRequired in boolean> <groupname for membership changes>"
	break
}

$InactiveDays = $args[0]
$detailsRequired = $args[1]
$OURequired = $args[2]
$GroupRequired = $args[3]
$GroupFilter = $args[4] 
try {
	Import-Module ActiveDirectory -ErrorAction Stop
}
catch
{
	Write-Host ("Either you do not have the ActiveDirectory module available on your system, or " +
	"your domain controllers do not have the Active Directory Web Services running.")
	break
}

try {
	Import-Module GroupPolicy -ErrorAction Stop
}
catch
{
	Write-host ("Group policy module is not available on your system.")
	break
}

$SearchBase = $(Get-ADRootDSE | Select-Object -ExpandProperty 'defaultNamingContext')
$domainInfo = Get-ADDomain -Current LocalComputer
$domain = $domainInfo.DistinguishedName

$ouProps = @('DistinguishedName','Name','whenChanged', 'whenCreated', 'CanonicalName')

Write-Host "Organizational Unit"
Write-Host "-------------------"
$ous = Get-ADOrganizationalUnit -SearchBase $SearchBase -Filter * -Properties $ouProps
Write-output "Number of OUs found: $($ous.count)"
if($detailsRequired -eq $true -and $OURequired -eq $true)
{
	if($($ous) -ne $null)
	{
		foreach($ou in $ous)
		{
			Write-host "$($ou.DistinguishedName)#~#$($ou.Name)#~#$($ou.WhenCreated)#~#$($ou.whenChanged)"
		}
	}
}

$emptyOuList = @()
if($($ous) -ne $null)
{
	foreach($ou in $ous)
	{
		$objectList = Get-ADObject -Filter * -SearchBase $ou.DistinguishedName -SearchScope OneLevel |
		Where-Object {$report.DistinguishedName -notcontains $_.DistinguishedName} |
		Select-Object -First 1
		if(-not $objectList)
		{
			$emptyOuList += $ou
		}
	}
}

write-host "Number of empty OUs found: $($emptyOuList.count)"
if($detailsRequired -eq $true)
{
	if($($emptyOuList) -ne $null)
	{
		foreach($ou in $emptyOuList)
		{
			Write-Host "$($ou.DistinguishedName)#~# $($ou.WhenCreated)"
		}
	}
}

Write-Host "Groups"
Write-Host "------"
$groupProps = @('DistinguishedName','Name','whenChanged', 'whenCreated', 'Members','isCriticalSystemObject')
$groups = Get-ADGroup -SearchBase $SearchBase -Filter * -Properties $groupProps
if($($groups) -ne $null)
{
	Write-host "Number of groups found: $($groups.count)"
}
if($detailsRequired -eq $true -and $GroupRequired -eq $true)
{
	if($($groups) -ne $null)
	{
		foreach($group in $groups)
		{
			Write-host "$($group.DistinguishedName)#~#$($group.Name)#~#$($group.WhenCreated)#~#$($group.whenChanged)"
		}
	}
}

$emptyGroupList = @()
if($($groups) -ne $null)
{
	foreach($group in $groups)
	{
		if((-not $group.isCriticalSystemObject) -and ($group.Members.Count -eq 0))
		{ 
			$emptyGroupList += $group
		}
	}
}
write-host "Number of empty groups found: $($emptyGroupList.count)"

if($detailsRequired -eq $true)
{
	if($($emptyGroupList) -ne $null)
	{
		foreach($group in $emptyGroupList)
		{
			#Write-Host "$($group.DistinguishedName)#~# $($group.WhenCreated)"
			Write-host "$($group.DistinguishedName)#~#$($group.Name)#~#$($group.WhenCreated)#~#$($group.whenChanged)"
		}
	}
}

Write-Host "Users"
Write-Host "-----"
$today_object = Get-Date
$unused_conditions_met = {
	## Ensure no built-in AD user objects are removed inadvertantly
	!$_.isCriticalSystemObject -and
	## The account is disabled (account cannot be used)
	(!$_.Enabled -or
	## The password is expired (account cannot be used)
	$_.PasswordExpired -or
	## The account has never been used
	!$_.LastLogonDate -or
	## The account hasn't been used for 60 days
	($_.LastLogonDate.AddDays($InactiveDays) -lt $today_object))
}

$unused_accounts = Get-ADUser -SearchBase $SearchBase -Filter * -Properties passwordexpired,lastlogondate,isCriticalSystemobject | Where-Object $unused_conditions_met

if($($unused_accounts) -ne $null)
{
	Write-Host "Number of Unused user accounts found: $($unused_accounts.count)"
}

if($detailsRequired -eq $true)
{
	if($($unused_accounts) -ne $null)
	{
		foreach($user in $unused_accounts)
		{
			if($user.LastLogonDate -eq $null)
			{
				$LastLoggedOnDaysAgo = '-5'
			}
			else
			{
				$LastLoggedOnDaysAgo = ($today_object - $user.LastLogonDate).Days
			}
			Write-Host "$($user.samAccountName)#~#$($user.Enabled)#~#$($user.PasswordExpired)#~#$LastLoggedOnDaysAgo"
		}
	}
}

Write-Host "GPOs"
Write-Host "----"
$disabledGPOs = @()
$gpos = Get-GPO -All -Domain $domain.DNSRoot
Write-Host "Number of GPOs found: $($gpos.count)"
if($($gpos) -ne $null)
{
	foreach($gpo in $gpos)
	{
		if($gpo.GpoStatus -like '*AllSettingsDisabled')
		{
			$disabledGPOs += $gpo
		}
	}
}

Write-Host "Number of disabled GPOs found: $($disabledGPOs.count)"
if($detailsRequired -eq $true)
{
	if($($disabledGPOs) -ne $null)
	{
		foreach($disabledGpo in $disabledGPOs)
		{
			$DisabledSettingsCategory = ([string]$disabledGpo.GpoStatus).TrimEnd('Disabled') 
			Write-host "$($disabledGpo.DisplayName)#~#$DisabledSettingsCategory#~#$($disabledGpo.Id)#~#" -NoNewline
			write-host "$($disabledGpo.Owner)#~#$($disabledGpo.ModificationTime)"
		}
	}
}

$emptyGPOs = @()
if($($gpos) -ne $null)
{
	foreach($gpo in $gpos)
	{
		if($gpo.Computer.DSVersion -eq 0 -and $gpo.User.DSVersion -eq 0)
		{
			$emptyGpos += $gpo
		}
	}
}

Write-host "Number of empty GPOs found: $($emptyGpos.count)"
if($detailsRequired -eq $true)
{
	if($($emptyGPOs) -ne $null)
	{
		foreach($emptyGpo in $emptyGPOs)
		{
			Write-host "$($emptyGpo.DisplayName)#~#$($disabledGpo.Id)#~#" -NoNewline
			write-host "$($disabledGpo.Owner)#~#$($disabledGpo.CreationTime)"
		}
	}
}

$UnlinkedGpos = @()
$GPOXmlObjs = @()
if($($gpos) -ne $null)
{
	foreach ($gpo in $gpos)
	{
		[xml]$oGpoReport = Get-GPOReport -Guid $gpo.ID -ReportType xml
		if ($($oGpoReport.GPO.LinksTo) -eq $null)
		{
			$UnlinkedGpos += $gpo
		}
		$GPOXmlObjs += $oGpoReport
	}
}

Write-host "Number of unlinked GPOs found: $($UnlinkedGpos.count)"
if($detailsRequired -eq $true)
{
	if($($UnlinkedGpos) -ne $null)
	{
		foreach($UnlinkedGpo in $UnlinkedGpos)
		{
			Write-host "$($UnlinkedGpo.DisplayName)#~#$($disabledGpo.Id)#~#" -NoNewline
			write-host "$($disabledGpo.Owner)#~#$($disabledGpo.CreationTime)"
		}
	}
}

$inactiveGpos = @() 
$inactiveGPos = $UnlinkedGpos + $disabledGPOs
Write-host "Number of inactive GPOs found: $($inactiveGpos.count)"

$aDefaultGpos = @('Default Domain Controllers Policy');
$NoSettingEnabledGpos = @()
if($($GPOXmlObjs) -ne $null)
{
	foreach($sGpo in $GPOXmlObjs)
	{
		$xGpo = ([xml]$sGpo).GPO;
		if ($aDefaultGpos -notcontains $xGpo.Name)
		{
			if (($xGpo.User.Enabled -eq 'true' -and ($($xGPO.User.ExtensionData)) -eq $null) -and `
			($xGpo.Computer.Enabled -eq 'true' -and ($($xGpo.Computer.ExtensionData)) -eq $null))
			{
				$NoSettingEnabledGpos += $xGPO
			}
		}
	}
}

$strCurrentTimeZone = (Get-WmiObject win32_timezone).StandardName
$TZ = [System.TimeZoneInfo]::FindSystemTimeZoneById($strCurrentTimeZone)

Write-Host "Number of GPOs with no settings enabled: $($NoSettingEnabledGpos.count)"
if($detailsRequired -eq $true)
{
	if($($NoSettingEnabledGpos) -ne $null)
	{
		foreach($xGPOObj in $NoSettingEnabledGpos)
		{
			$LocalTime = [System.TimeZoneInfo]::ConvertTimeFromUtc($xGPOObj.CreatedTime, $TZ)
			Write-host "$($xGPOObj.Name)#~#$($LocalTime)"
		}
	}
}

$dc_to_use = $(Get-ADRootDSE | Select-Object -ExpandProperty 'dnsHostName')
If ($dc_to_use -eq $null )
{
	Write-Host "Cannot reach $DomainFQDN."
	break
}

If ($GroupFilter -eq $null -or $GroupFilter -eq "")
{
	$searcher_filter = "(&(objectClass=group)(adminCount=1))" 
}
Else
{
	$searcher_filter = "(&(objectClass=group)(name=$GroupFilter))"
}

$today = Get-Date
$config_nc = [string] ([ADSI]"LDAP://$dc_to_use/RootDSE").configurationNamingContext
$TSL_ADSI = [string] ([ADSI]"LDAP://$dc_to_use/CN=Directory Service,CN=Windows NT,CN=Services,$_config_nc").tombstoneLifetime 
If ( $TSL_ADSI -eq "" )
{
	$threshold = 60
}
Else
{ 
	$threshold = $($_TSL_ADSI)
}
$output_obj = @()
$searcher_root = [ADSI]"LDAP://$dc_to_use"
$searcher_properties = "sAMAccountName","msDS-ReplValueMetaData","distinguishedName"
$searcher = New-Object System.DirectoryServices.DirectorySearcher( $searcher_root , $searcher_filter , $searcher_properties )
$searcher.PageSize = 1000
$searcher_results = $searcher.FindAll()
$searcher_results_count = $searcher_results.Count
if($searcher_results_count -eq 0)
{
	Write-Host "Number of group membership changed: 0"
	break
}
$searcher_results | Sort-Object -Property Path | ForEach-Object `
{
	$current_dn                = [string] $_.Properties.distinguishedname
	$current_samaccountname    = [string] $_.Properties.samaccountname
	$current_replvaluemetadata = $_.Properties."msds-replvaluemetadata"
	$current_replvaluemetadata | ForEach-Object `
	{
		# Store the current value in a XML object
		$current_metadata = $_
		# In case we have an ampersand, we escape it to be able to convert the object to XML
		If ( $_current_metadata -ne $null )
		{
			$current_metadata = $current_metadata.Replace("&","&amp;")
		}
		# Convert into XML
		$current_metadata_xml = [XML] $current_metadata
		# For each item in the XML model we read the child
		$current_metadata_xml | ForEach-Object `
		{
			# Check if the linked value is a member attribute
			If ( $_.DS_REPL_VALUE_META_DATA.pszAttributeName -eq "member" )
			{
				$pszObjectDn   = $_.DS_REPL_VALUE_META_DATA.pszObjectDn
				$dwVersion     = $_.DS_REPL_VALUE_META_DATA.dwVersion
				$ftimeDeleted  = $_.DS_REPL_VALUE_META_DATA.ftimeDeleted
				$ftimeCreated  = $_.DS_REPL_VALUE_META_DATA.ftimeCreated
				#$_ftimeLastOriginatingChange = $_.DS_REPL_VALUE_META_DATA.ftimeLastOriginatingChange #No used yet
				# Check if the member has been added or removed before the threshold
				If ( ([datetime] $ftimeDeleted) -ge $today.AddDays(-$threshold) -or ([datetime] $ftimeCreated) -ge $today.AddDays(-$threshold) )
				{
					# Check if the membership is still valid
					If ( $ftimeDeleted -ne "1601-01-01T00:00:00Z")
					{
						$flag_operation = "removed"
						$datemodified   = $ftimeDeleted        
					}
					Else
					{
						$flag_operation = "added"
						$datemodified   = $ftimeCreated
					}
					$LocalTime = [System.TimeZoneInfo]::ConvertTimeFromUtc($ftimeCreated, $TZ)
					$line = "$($current_dn)#~#$($current_samaccountname)#~#$($pszObjectDn)#~#$($flag_operation)#~#$($LocalTime )#~#$($datemodified)#~#$($dwVersion)"
					$output_obj += $line
				}
			}
		}
	}
}
Write-Host "Number of group membership changed: $($output_obj.count)"
if($detailsRequired -eq $true)
{
	if($($output_obj) -ne $null)
	{
		foreach($obj in $output_obj)
		{
			Write-host "$($obj)"
		}
	}
}



