'use strict'

const workerThread = require('worker_threads');

const logger = require('../logger');
const Queue = require('./queue');
const Socket = require('./socket-client');
const utils = require('./collector-utils');
const Config = require('./../default_config');

let config = {};
let timer = null;
let queue = null;
let socket = null;
let isStarted = false;
const appendMsg = `Socket-thread=${workerThread.threadId}:`;

logger.info(appendMsg, 'started successfully..!');

workerThread.parentPort.on('message', (message) => {
  if (message && message.action === 'startSocket') {
    config = message.config;
    _start();
    return;
  }

  if (!isStarted) return;
  if (message === 'stopSocket') {
    _stopTimer();
    socket.stop();
  } else if (message === 'resetConnection') {
    socket.resetConnection();
  } else if (message === 'setTimeout') {
    socket.setTimeout();
  } else if (message && message.action === 'setConfig' && message.config) {
    socket.setConfig(message.config);
  } else if (message && message.action === 'add' && message.metrics) {
    const data = message.metrics;
    if (data.sendImmediately) {
      if (!socket || !socket.isConnected) return;
      if (data.type === 'String' && data.message) {
        socket.send(data.message, true, true);
      } else {
        delete data.sendImmediately;
        socket.send([data], true);
      }
      return;
    }

    queue.enqueue(data);

    /** if no batch size is given then the app will send one by one immediattly  */
    if (!config.max_batch_size) {
      _sendData();
    }
  } else if (message && message.action === 'sendFile' && message.type && message.payload) {
    socket.sendFile(message.type, message.payload);
  }
});

function _start() {
  logger.setConfig(config);

  if (!queue) {
    queue = new Queue(config);
    socket = new Socket(config);
    socket.start(config);

    socket.event.on('stopTracing', _ => {
      logger.debug(appendMsg, `${queue.size()} data is going to discard`);
      queue.reset();
      workerThread.parentPort.postMessage({ action: 'stopTracing' });
      _stopTimer.call(this);
    });

    socket.event.on('startTracing', _ => {
      workerThread.parentPort.postMessage({ action: 'startTracing' });
      _startTimer.call(this);
    });

    socket.event.on('configChanged', newConfig => {
      workerThread.parentPort.postMessage({
        action: 'configChanged',
        config: newConfig
      });
    });

    socket.event.on('onNewComponent', newComponentDetails => {
      workerThread.parentPort.postMessage({
        action: 'newComponentDetails',
        data: newComponentDetails
      });
    });
  }

  _startTimer();
  isStarted = true;
}

function _startTimer() {
  if (timer) return;
  const interval = config.flush_interval || Config.flush_interval;
  timer = setInterval(() => {
    if (queue.size() <= 0) return;
    _sendData();
  }, interval);

  timer.unref();
}

function _stopTimer() {
  if (timer) {
    clearInterval(timer);
    timer = null;
  }
}

function _sendData() {
  utils.sendDataToSocket(socket, queue, config, appendMsg);
}